<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Solidres\Media;

defined('_JEXEC') or die;

use Joomla\CMS\Factory;
use Joomla\CMS\Access\Exception\NotAllowed;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Image\Image;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Session\Session;
use Joomla\Input\Json;
use Joomla\Registry\Registry;
use RuntimeException, SRPlugin, SRUtilities;
use Throwable;

trait ImageUploaderTrail
{
	protected function checkPermission(string $type, int $id)
	{
		if (!in_array($type, ['PROPERTY', 'ROOM_TYPE', 'EXPERIENCE']))
		{
			throw new RuntimeException('Invalid request type');
		}

		$db    = Factory::getDbo();
		$query = $db->getQuery(true);

		switch ($type)
		{
			case 'PROPERTY';
				$query->select('a.id')
					->from($db->quoteName('#__sr_reservation_assets', 'a'))
					->where('a.id = ' . $id);
				break;

			case 'ROOM_TYPE';
				$query->select('a.reservation_asset_id AS id')
					->from($db->quoteName('#__sr_room_types', 'a'))
					->where('a.id = ' . $id);
				break;

			case 'EXPERIENCE';
				if (SRPlugin::isEnabled('user'))
				{
					$query->select('a.id, a2.user_id AS userId')
						->from($db->quoteName('#__sr_experiences', 'a'))
						->leftJoin($db->quoteName('#__sr_customers', 'a2') . ' ON a2.id = a.partner_id')
						->where('a.id = ' . $id);
				}
				else
				{
					$query->select('a.id')
						->from($db->quoteName('#__sr_experiences', 'a'))
						->where('a.id = ' . $id);
				}

				break;
		}

		if (!($item = $db->setQuery($query)->loadObject()))
		{
			throw new RuntimeException('Invalid request ID');
		}

		$app = Factory::getApplication();

		if ($app->isClient('site'))
		{
			$access = false;

			if ('EXPERIENCE' === $type)
			{
				$access = !empty($item->userId) && $item->userId == $app->getIdentity()->id;
			}
			else
			{
				if ($assets = SRUtilities::getPropertiesByPartner())
				{
					foreach ($assets as $asset)
					{
						if ($asset->id == $item->id)
						{
							$access = true;
							break;
						}
					}
				}
			}

			if (!$access)
			{
				throw new NotAllowed(Text::_('JERROR_ALERTNOAUTHOR'));
			}
		}
	}

	protected function extractMediaInputs($app)
	{
		$resourceData = [];

		foreach ($app->input->get('SRMediaResourceData', [], 'ARRAY') as $data)
		{
			if (
				is_string($data)
				&& ($decode = json_decode($data, true))
				&& is_string($decode['name'] ?? null)
				&& is_string($decode['type'] ?? null)
				&& is_integer($decode['noThumb'] ?? null)
			)
			{
				$decode['files'] = $app->input->files->get(($decode['name'] ? str_replace('.', '_', $decode['name']) . '_' : '') . 'SRUploadedMedia', [], 'ARRAY');
				$resourceData[]  = $decode;
			}
		}

		return $resourceData;
	}

	public function loadResources()
	{
		$app = Factory::getApplication();

		try
		{
			$input = $this->extractMediaInputs($app)[0] ?? [];

			if (empty($input))
			{
				throw new RuntimeException('Invalid request.');
			}

			$id = $app->input->getUint('id', 0);
			$this->checkPermission($input['type'], $id);

			if ($sources = ($id ? ImageUploaderHelper::getData($id, $input['type']) : []))
			{
				foreach ($sources as &$source)
				{
					$source = ImageUploaderHelper::getImageThumb($source, ImageUploaderHelper::getThumbByType($input['type']), true);
				}
			}

			echo json_encode(['success' => true, 'data' => $sources]);
		}
		catch (Throwable $e)
		{
			echo json_encode(['success' => false, 'message' => $e->getMessage()]);
		}

		$app->close();
	}

	public function reOrderResources()
	{
		$app = Factory::getApplication();

		try
		{
			if (!Session::checkToken())
			{
				throw new RuntimeException(Text::_('JINVALID_TOKEN'));
			}

			$input = $this->extractMediaInputs($app)[0] ?? [];

			if (empty($input))
			{
				throw new RuntimeException('Invalid request');
			}

			$id = $app->input->get('id', 0, 'UINT');
			$this->checkPermission($input['type'], $id);
			$sources   = (new Json())->get('sources', [], 'ARRAY');
			$tableMaps = [
				'property'   => '#__sr_reservation_assets',
				'room_type'  => '#__sr_room_types',
				'experience' => '#__sr_experiences',
			];

			if ($tbl = ($tableMaps[strtolower($input['type'])] ?? null))
			{
				$db    = Factory::getDbo();
				$query = $db->getQuery(true)
					->update($db->quoteName($tbl))
					->set($db->quoteName('images') . ' = ' . $db->quote(json_encode($sources)))
					->where($db->quoteName('id') . ' = ' . $db->quote($id));
				$db->setQuery($query)
					->execute();
			}

			$subPath = strtolower($input['type'])[0];
			echo json_encode([
				'success' => true,
				'data'    => array_map(function ($source) use ($input, $subPath) {
					return ImageUploaderHelper::getImageThumb($subPath . '/' . $source, ImageUploaderHelper::getThumbByType($input['type']), true);
				}, $sources),
			]);
		}
		catch (Throwable $e)
		{
			echo json_encode(['success' => false, 'message' => $e->getMessage()]);
		}

		$app->close();
	}

	public function uploadMedia(int $id)
	{
		$app        = Factory::getApplication();
		$thumbSizes = ImageUploaderHelper::getThumbSizes();
		$db         = Factory::getDbo();
		$results    = [];

		foreach ($this->extractMediaInputs($app) as $input)
		{
			$this->checkPermission($input['type'], $id);
			$name       = strtoupper($input['type'] . '.' . $id . ($input['name'] ? '.' . $input['name'] : ''));
			$subPath    = strtolower($input['type'])[0] . '/' . $id;
			$uploadPath = ImageUploaderHelper::getUploadPath() . '/' . $subPath;
			$messages   = [];
			$sources    = [];

			if ($input['files'])
			{
				if (!is_dir($uploadPath))
				{
					Folder::create($uploadPath);
				}

				foreach ($input['files'] as $file)
				{
					if (0 !== strpos($file['type'], 'image/'))
					{
						$messages[] = Text::sprintf('SR_MEDIA_ERROR_NOT_IMAGE', $file['name']);
						continue;
					}

					$ext      = explode('.', $file['name'])[1];
					$fileBase = md5($id . ':' . $file['name'] . ':' . uniqid());
					$fileName = $fileBase . '.' . $ext;

					if (!empty($file['error']) || !File::upload($file['tmp_name'], $uploadPath . '/' . $fileName))
					{
						$messages[] = Text::sprintf('SR_MEDIA_FILE_UPLOAD_ERROR', $file['name']);
						continue;
					}

					if (!$input['noThumb'])
					{
						$jImage = new Image($uploadPath . '/' . $fileName);
						$jImage->createThumbnails($thumbSizes, Image::CROP_RESIZE);
					}

					$sources[] = $fileName;
				}
			}

			$results[] = [
				'sources'  => array_map(
					function ($source) use ($input, $subPath) {
						return ImageUploaderHelper::getImageThumb($subPath . '/' . $source, ImageUploaderHelper::getThumbByType($input['noThumb'] ? '' : $input['type']), true);
					},
					$sources
				),
				'messages' => $messages,
			];

			try
			{
				// Store params data
				[$type, $recordId, $paramsName, $paramName] = $this->extractMediaParamName($name);
				$source    = $sources[0] ?? null;
				$tableMaps = [
					'property'   => '#__sr_reservation_assets',
					'room_type'  => '#__sr_room_types',
					'experience' => '#__sr_experiences',
				];
				$tbl       = $tableMaps[$type] ?? null;

				if ($source
					&& $recordId
					&& $paramsName
					&& $paramName
					&& $tbl
				)
				{
					$recordId = (int) $recordId;
					$registry = new Registry;
					$query    = $db->getQuery(true)
						->select('a.' . $paramsName)
						->from($db->quoteName($tbl, 'a'))
						->where('a.id = ' . $recordId);

					if ($paramsData = $db->setQuery($query)->loadResult())
					{
						$registry->loadString($paramsData);
					}

					$registry->set($paramName, $source);
					$query->clear()
						->update($db->quoteName($tbl))
						->set($db->quoteName($paramsName) . ' = ' . $db->quote($registry->toString()))
						->where($db->quoteName('id') . '=' . $recordId);
					$db->setQuery($query)
						->execute();
				}
				elseif ($id && $sources)
				{
					$id    = (int) $id;
					$query = $db->getQuery(true)
						->select('a.images')
						->from($db->quoteName($tbl, 'a'))
						->where('a.id = ' . $id);

					if ($images = $db->setQuery($query)->loadResult())
					{
						$sources = array_merge(json_decode($images, true), $sources);
					}

					$query->clear()
						->update($db->quoteName($tbl))
						->set($db->quoteName('images') . '=' . $db->quote(json_encode($sources)))
						->where($db->quoteName('id') . '=' . $id);
					$db->setQuery($query)
						->execute();
				}
			}
			catch (RuntimeException $e)
			{
			}
		}

		return $results;
	}

	public function uploadResources()
	{
		$app = Factory::getApplication();

		try
		{
			if (!Session::checkToken())
			{
				throw new RuntimeException(Text::_('JINVALID_TOKEN'));
			}

			$id = $app->input->getUint('id', 0);

			echo json_encode(['success' => true, 'data' => $this->uploadMedia($id)]);
		}
		catch (Throwable $e)
		{
			echo json_encode(['success' => false, 'message' => $e->getMessage()]);
		}

		$app->close();
	}

	private function extractMediaParamName(string $name)
	{
		$parts      = explode('.', strtolower($name), 4);
		$type       = $parts[0] ?? '';
		$id         = $parts[1] ?? null;
		$paramsName = $parts[2] ?? null;
		$paramName  = $parts[3] ?? null;

		return [$type, $id, $paramsName, $paramName];
	}

	public function removeResource()
	{
		$app = Factory::getApplication();

		try
		{
			$input = $this->extractMediaInputs($app)[0] ?? [];

			if (empty($input))
			{
				throw new RuntimeException('Invalid request.');
			}

			$json = new Json();
			$id   = $app->input->get('id', 0, 'UINT');
			$src  = $json->getString('src', '');
			$this->checkPermission($input['type'], $id);
			$db = Factory::getDbo();
			$name = strtoupper($input['type'] . '.' . $id . ($input['name'] ? '.' . $input['name'] : ''));
			[$type, $recordId, $paramsName, $paramName] = $this->extractMediaParamName($name);
			$tableMaps = [
				'property'   => '#__sr_reservation_assets',
				'room_type'  => '#__sr_room_types',
				'experience' => '#__sr_experiences',
			];
			$tbl = $tableMaps[$type] ?? null;

			if ($sources = ImageUploaderHelper::getData($id, $input['type']))
			{
				foreach ($sources as $i => $source)
				{
					if (ImageUploaderHelper::getImage($source, 'full', true) === $src)
					{
						ImageUploaderHelper::removeFullResource($source);
						unset($sources[$i]);

						if ($tbl)
						{
							$sources = json_encode(
								array_map(
									function ($source) {
										return basename($source);
									},
									array_values($sources)
								)
							);
							$query   = $db->getQuery(true)
								->update($db->quoteName($tbl))
								->set($db->quoteName('images') . ' = ' . $db->quote($sources))
								->where($db->quoteName('id') . ' = ' . $id);
							$db->setQuery($query)
								->execute();
						}

						break;
					}
				}
			}

			if ($paramsName
				&& $paramName
				&& $tbl
			)
			{
				$registry = new Registry;
				$recordId = (int) $recordId;
				$query    = $db->getQuery(true)
					->select('a.' . $paramsName)
					->from($db->quoteName($tbl, 'a'))
					->where('a.id = ' . $recordId);

				if ($paramsData = $db->setQuery($query)->loadResult())
				{
					$registry->loadString($paramsData);

					if (($source = $registry->get($paramName)) && $source === basename($src))
					{
						ImageUploaderHelper::removeFullResource($source);
					}

					$registry->set($paramName, '');
					$query->clear()
						->update($db->quoteName($tbl))
						->set($db->quoteName($paramsName) . ' = ' . $db->quote($registry->toString()))
						->where($db->quoteName('id') . ' = ' . $recordId);
					$db->setQuery($query)
						->execute();
				}
			}

			echo json_encode(['success' => true, 'data' => $src]);
		}
		catch (Throwable $e)
		{
			echo json_encode(['success' => false, 'message' => $e->getMessage()]);
		}

		$app->close();
	}
}