<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Solidres\Media;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Filesystem\File;
use Joomla\CMS\Filesystem\Folder;
use Joomla\CMS\Uri\Uri;
use Joomla\Registry\Registry;
use RuntimeException;

defined('_JEXEC') or die;

class ImageUploaderHelper
{
	public static function getData(int $id, string $type)
	{
		$tableMaps = [
			'property'   => '#__sr_reservation_assets',
			'room_type'  => '#__sr_room_types',
			'experience' => '#__sr_experiences',
		];

		if ($tbl = $tableMaps[strtolower($type)] ?? null)
		{
			$db    = Factory::getDbo();
			$query = $db->getQuery(true)
				->select('a.images')
				->from($db->quoteName($tbl, 'a'))
				->where('a.id = ' . $db->quote($id));

			if ($images = $db->setQuery($query)->loadResult())
			{
				$sub = strtolower($type)[0] . '/' . $id;

				return array_map(function ($image) use ($sub) {
					return $sub . '/' . $image;
				}, json_decode($images, true));
			}
		}

		return [];
	}

	public static function getImage(string $source, string $size = 'full', $relative = false)
	{
		$uploadBasePath = static::getUploadPath(true);
		$hostUrl        = $relative ? 'images/' . $uploadBasePath : Uri::root() . 'images/' . $uploadBasePath;
		$solidresParams = ComponentHelper::getParams('com_solidres');
		$displaySize    = [
			'asset_small'     => $solidresParams->get('asset_thumb_small', '75x75'),
			'asset_medium'    => $solidresParams->get('asset_thumb_medium', '300x250'),
			'asset_large'     => $solidresParams->get('asset_thumb_large', '875x350'),
			'roomtype_small'  => $solidresParams->get('roomtype_thumb_small', '75x75'),
			'roomtype_medium' => $solidresParams->get('roomtype_thumb_medium', '300x250'),
			'roomtype_large'  => $solidresParams->get('roomtype_thumb_large', '875x350'),
		];

		if (isset($displaySize[$size]))
		{
			$parts        = explode('/', $source);
			[$name, $ext] = explode('.', array_pop($parts));
			$sub          = ($parts ? '/' . implode('/', $parts) : '');

			return $hostUrl . $sub . '/thumbs/' . $name . '_' . $displaySize[$size] . '.' . $ext;
		}

		return $hostUrl . '/' . $source;
	}

	public static function getThumbSizes()
	{
		$thumbNewSizes = [];
		$thumbSizes    = preg_split('/\r\n|\n|\r/', ComponentHelper::getParams('com_solidres')->get('thumb_sizes', ''));

		// Validate sizes
		for ($tid = 0, $tCount = count($thumbSizes); $tid < $tCount; $tid++)
		{
			if (empty($thumbSizes[$tid]) || ctype_space($thumbSizes[$tid]))
			{
				continue;
			}

			$thumbNewSizes[] = trim($thumbSizes[$tid]);
		}

		if (!$thumbNewSizes)
		{
			$thumbNewSizes = ['300x250', '75x75'];
		}

		return $thumbNewSizes;
	}

	private static function _migrate(array $list, array $thumbSizes, string $tbl, &$removedFiles)
	{
		$tempPath = static::getUploadPath();
		$sources  = [];

		switch ($tbl)
		{
			case '#__sr_reservation_assets':
				$tempPath .= '/p';
				break;

			case '#__sr_room_types':
				$tempPath .= '/r';
				break;

			case '#__sr_experiences':
				$tempPath .= '/e';
				break;
		}

		foreach ($list as $media)
		{
			$image = SRPATH_MEDIA_IMAGE_SYSTEM . '/' . $media->value;

			if (!is_file($image))
			{
				continue;
			}

			$path = $tempPath . '/' . $media->targetId;

			if (!is_dir($path . '/thumbs'))
			{
				Folder::create($path . '/thumbs');
			}

			[$name, $ext] = explode('.', $media->value);
			$thumbs = [];

			foreach ($thumbSizes as $thumbSize)
			{
				$imagePath = SRPATH_MEDIA_IMAGE_SYSTEM . '/thumbnails/' . $name . '_' . $thumbSize . '.' . $ext;

				if (is_file($imagePath))
				{
					$thumbs[] = [
						'src'  => $imagePath,
						'dest' => $path . '/thumbs/' . basename($imagePath),
					];
				}
			}

			// B/C
			foreach (['thumbnails/1', 'thumbnails/2'] as $thumbDir)
			{
				$imagePath = SRPATH_MEDIA_IMAGE_SYSTEM . '/' . $thumbDir . '/' . $name . '.' . $ext;

				if (is_file($imagePath))
				{
					$thumbs[] = [
						'src'  => $imagePath,
						'dest' => $path . '/thumbs/' . $name . '_' . ($thumbDir === 'thumbnails/1' ? '300x250' : '75x75') . '.' . $ext,
					];
				}
			}

			if (!isset($sources[$media->targetId]))
			{
				$sources[$media->targetId] = [];
			}

			$sources[$media->targetId][] = [
				'image'   => [
					'src'  => $image,
					'dest' => $path . '/' . $media->value,
				],
				'thumbs' => $thumbs,
			];
		}

		if ($sources)
		{
			$db = Factory::getDbo();

			foreach ($sources as $targetId => $media)
			{
				$images = [];

				foreach ($media as $source)
				{
					if (File::copy($source['image']['src'], $source['image']['dest']))
					{
						$images[]       = basename($source['image']['dest']);
						$removedFiles[] = $source['image']['src'];

						if ($source['thumbs'])
						{
							foreach ($source['thumbs'] as $thumb)
							{
								if (File::copy($thumb['src'], $thumb['dest']))
								{
									$removedFiles[] = $thumb['src'];
								}
							}
						}
					}
				}

				$query = $db->getQuery(true)
					->update($db->quoteName($tbl))
					->set($db->quoteName('images') . ' = ' . $db->quote(json_encode($images)))
					->where($db->quoteName('id') . ' = ' . $db->quote($targetId));
				$db->setQuery($query)
					->execute();
			}
		}
	}

	public static function migrate()
	{
		$db           = Factory::getDbo();
		$removedFiles = [];
		$thumbSizes   = static::getThumbSizes();

		// Migrate for property
		$query = $db->getQuery(true)
			->select('a.value, a2.reservation_asset_id AS targetId')
			->from($db->quoteName('#__sr_media', 'a'))
			->join('INNER', $db->quoteName('#__sr_media_reservation_assets_xref', 'a2') . ' ON a2.media_id = a.id')
			->order('a2.weight ASC');

		if ($list = $db->setQuery($query)->loadObjectList())
		{
			static::_migrate($list, $thumbSizes, '#__sr_reservation_assets', $removedFiles);
		}

		$query->clear()
			->select('a.id, a.params')
			->from($db->quoteName('#__sr_reservation_assets', 'a'));

		if ($assets = $db->setQuery($query)->loadObjectList())
		{
			foreach ($assets as $asset)
			{
				$registry = new Registry($asset->params);

				if ($logo = $registry->get('logo'))
				{
					$oldPath = SRPATH_MEDIA_IMAGE_SYSTEM . '/' . $logo;
					$file    = basename($logo);
					$path    = static::getUploadPath() . '/p/' . $asset->id . '/' . $file;

					if (!is_file($path) && is_file($oldPath))
					{
						File::copy($oldPath, $path);
						$registry->set('logo', $file);
						$query->clear()
							->update($db->quoteName('#__sr_reservation_assets'))
							->set($db->quoteName('params') . ' = ' . $db->quote($registry->toString()))
							->where($db->quoteName('params') . ' = ' . $asset->id);
						$db->setQuery($query)
							->execute();
					}
				}
			}
		}

		// Migrate for room types
		$query->clear()
			->select('a.value, a2.room_type_id AS targetId')
			->from($db->quoteName('#__sr_media', 'a'))
			->join('INNER', $db->quoteName('#__sr_media_roomtype_xref', 'a2') . ' ON a2.media_id = a.id')
			->order('a2.weight ASC');

		if ($list = $db->setQuery($query)->loadObjectList())
		{
			static::_migrate($list, $thumbSizes, '#__sr_room_types', $removedFiles);
		}

		if ($removedFiles)
		{
			foreach (array_unique($removedFiles) as $removedFile)
			{
				File::delete($removedFile);
			}
		}
	}

	public static function getUploadPath($base = false)
	{
		static $uploadBasePath = '';

		if (!$uploadBasePath)
		{
			$uploadBasePath = ComponentHelper::getParams('com_solidres')->get('images_storage_path', 'bookingengine');
			$uploadBasePath = Folder::makeSafe(str_replace('/\/+/', '/', trim($uploadBasePath, '/')));
		}

		return $base ? $uploadBasePath : JPATH_ROOT . '/images/' . $uploadBasePath;
	}

	public static function getImageThumb(string $source, string $size = 'full', $relative = false)
	{
		return [
			'image' => static::getImage($source, 'full', $relative),
			'thumb' => static::getImage($source, $size, $relative),
		];
	}

	public static function removeFullResource(string $source)
	{
		$parts        = explode('/', $source);
		[$name, $ext] = explode('.', array_pop($parts));
		$uploadPath = static::getUploadPath();
		$fullPath   = $uploadPath . '/' . $source;

		if (is_file($fullPath))
		{
			File::delete($fullPath);
		}

		$sub = $parts ? '/' . implode('/', $parts)  : '';

		foreach (static::getThumbSizes() as $thumbSize)
		{
			$thumb = $uploadPath . $sub . '/thumbs/' . $name . '_' . $thumbSize . '.' . $ext;

			if (is_file($thumb))
			{
				File::delete($thumb);
			}
		}
	}

	public static function getThumbByType(string $type)
	{
		$thumbsMap = [
			'property'  => 'asset_medium',
			'room_type' => 'roomtype_medium',
		];

		return $thumbsMap[strtolower($type)] ?? 'full';
	}
}