<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Solidres\Plugin;

defined('_JEXEC') or die;

/**
 * The base payment class for Solidres's payment implementation
 *
 * @package       Solidres
 * @subpackage    Payment
 * @since         0.5.0
 */

use Joomla\CMS\Form\Form;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Table\Table;
use Joomla\CMS\Uri\Uri;
use Joomla\Event\DispatcherInterface;
use Joomla\Registry\Registry;
use Joomla\CMS\Log\Log;
use RuntimeException;
use Solidres\Config\Config;
use Joomla\Component\Solidres\Administrator\Helper\SolidresHelper;

class PaymentPlugin extends Plugin
{
    public string $identifier = "base";

    public string $title = "base";

    public string $description = "";

    public array $acceptedCurrencies = [];

    public string $image = "";

    public int $defaultState;

    public int $confirmationState;

    public int $confirmationPaymentState;

    public int $pendingState;

    public int $pendingPaymentState;

    public int $cancellationState;

    public int $cancellationPaymentState;

    public float $amountToBePaid;

    public string $countryCode2;

    public string $countryStateCode2;

    protected array $dataConfig;

    public function __construct(DispatcherInterface $dispatcher, array $config = [])
    {
        parent::__construct($dispatcher, $config);
        $this->identifier               = $this->_name;
        $this->title                    = Text::_('SR_PAYMENT_METHOD_' . strtoupper($this->identifier));
        $this->description              = Text::_(
            'SR_PAYMENT_METHOD_' . strtoupper($this->identifier) . '_DESCRIPTION'
        );
        $this->defaultState             = $this->solidresConfig->get('default_reservation_state', 0);
        $this->confirmationState        = $this->solidresConfig->get('confirm_state', 5);
        $this->confirmationPaymentState = $this->solidresConfig->get('confirm_payment_state', 1);
        $this->pendingState             = $this->solidresConfig->get('pending_state', 0);
        $this->pendingPaymentState      = $this->solidresConfig->get('pending_payment_state', 3);
        $this->cancellationState        = $this->solidresConfig->get('cancel_state', 4);
        $this->cancellationPaymentState = $this->solidresConfig->get('cancel_payment_state', 2);

        static $log = null;

        if ($log == null) {
            $options['format']    = '{DATE}\t{TIME}\t{LEVEL}\t{CODE}\t{MESSAGE}';
            $options['text_file'] = 'solidres_' . $this->identifier . '.php';
            Log::addLogger($options, Log::DEBUG, [$this->identifier]);
            $log = true;
        }
    }

    public function onSolidresPaymentNew($reservationData)
    {
        $this->log('Start ' . $this->title . ' payment processing for reservation id ' . $reservationData->id);
        $tableCountry = $this->createTable('Country');
        $tableState   = $this->createTable('State');
        $tableCountry->load($reservationData->customer_country_id);
        $tableState->load($reservationData->customer_geo_state_id);
        $this->dataConfig        = $this->loadFormData((int)$reservationData->reservation_asset_id);
        $this->amountToBePaid    = $this->getPayAmount($reservationData);
        $this->countryCode2      = $tableCountry->code_2;
        $this->countryStateCode2 = $tableState->code_2 ?? '';
    }

    public function onSolidresPaymentCallback($paymentMethodId, $callbackData)
    {
    }

    protected function getNotifyUrl($parameters = [])
    {
        return Uri::root(
            ) . 'index.php?option=com_solidres&task=reservation.paymentcallback&payment_method_id=' . $this->identifier . (!empty($parameters) ? '&' . http_build_query(
                    $parameters
                ) : '');
    }

    protected function getReturnUrl($reservationId, $parameters = [])
    {
        return Uri::root(
            ) . 'index.php?option=com_solidres&task=reservation.finalize&reservation_id=' . $reservationId . (!empty($parameters) ? '&' . http_build_query(
                    $parameters
                ) : '');
    }

    protected function getCancelUrl($assetId)
    {
        $app          = $this->getApplication();
        $context      = 'com_solidres.reservation.process';
        $activeItemId = $app->getUserState($context . '.activeItemId');

        $url = Route::_(
            'index.php?option=com_solidres&view=reservationasset&id=' . $assetId . ($activeItemId ? '&Itemid=' . $activeItemId : ''),
            false
        );
        $url = trim(Uri::getInstance()->toString(['host', 'scheme']) . $url);

        return $url;
    }

    public function onReservationFinalize($context, &$reservationId)
    {
        $app = $this->getApplication();

        // Retrieve payment method id
        $guestInfo = $app->getUserState($context . '.guest');

        if ($guestInfo['payment_method_id'] != $this->identifier) {
            return false;
        }

        return true;
    }

    public function getPayAmount($reservationData)
    {
        if ((float)$reservationData->deposit_amount > 0) {
            $amountToBePaid = (float)$reservationData->deposit_amount;
        } else {
            if ($reservationData->discount_pre_tax == 1) {
                $amountToBePaid = $reservationData->total_price_tax_excl - $reservationData->total_discount + $reservationData->tax_amount + $reservationData->total_extra_price_tax_incl;
            } else {
                $amountToBePaid = $reservationData->total_price_tax_excl + $reservationData->tax_amount - $reservationData->total_discount + $reservationData->total_extra_price_tax_incl;
            }

            if ($reservationData->tourist_tax_amount > 0) {
                $amountToBePaid += $reservationData->tourist_tax_amount;
            }
        }

        if ($reservationData->payment_method_surcharge > 0) {
            $amountToBePaid += $reservationData->payment_method_surcharge;
        }

        if ($reservationData->payment_method_discount > 0) {
            $amountToBePaid -= $reservationData->payment_method_discount;
        }

        return $amountToBePaid;
    }

    public function onReservationAssetPrepareForm($form, $data)
    {
        if ($this->solidresConfig->get('frontend_payment_method_manage', 1) == 0 && $this->getApplication()->isClient(
                'site'
            )) {
            return;
        }

        $this->loadLanguage();

        if (!($form instanceof Form)) {
            throw new RuntimeException(Text::_('JERROR_NOT_A_FORM'));
        }

        // Check we are manipulating a valid form.
        if (!in_array($form->getName(), ['com_solidres.reservationasset'])) {
            return true;
        }

        if (method_exists($this, 'getForm')) {
            $file = $this->getForm();
        } elseif (file_exists(JPATH_PLUGINS . '/solidrespayment/' . $this->_name . '/forms/' . $this->_name . '.xml')) {
            $file = JPATH_PLUGINS . '/solidrespayment/' . $this->_name . '/forms/' . $this->_name . '.xml';
        }

        $arrayData = (array)$data;

        if (isset($file)
            && $form->loadFile($file, false)
            && !empty($arrayData['id'])) {
            $form->bind(['payments' => $this->loadFormData($arrayData['id'])]);
        }

        return true;
    }

    protected function loadFormData($scopeId = 0)
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true)
            ->select('a.data_key, a.data_value')
            ->from($db->quoteName('#__sr_config_data', 'a'))
            ->where('a.data_key LIKE ' . $db->quote('payments/' . $this->identifier . '/%'))
            ->where('a.scope_id = ' . (int)$scopeId);
        $db->setQuery($query);

        if ($data = $db->loadObjectList('data_key')) {
            $temp = [];

            foreach ($data as $k => $v) {
                $k        = str_replace('payments/' . $this->identifier . '/', '', $k);
                $temp[$k] = $v->data_value;

                if (is_string($temp[$k])
                    && is_array(json_decode($temp[$k], true))
                    && (json_last_error() == JSON_ERROR_NONE)
                ) {
                    $temp[$k] = json_decode($temp[$k], true);
                }
            }

            $data = $temp;
        }

        return $data;
    }

    protected function getConfig($scopeId = 0)
    {
        $options = ['data_namespace' => 'payments/' . $this->identifier];

        if ($scopeId > 0) {
            $options['scope_id'] = (int)$scopeId;
        }

        $config = new Config($options);

        return $config;
    }

    public function onReservationAssetAfterSave($data, $table, $result, $isNew)
    {
        if ($this->solidresConfig->get('frontend_payment_method_manage', 1) == 0 && $this->getApplication()->isClient(
                'site'
            )) {
            return;
        }

        if (method_exists($this, 'getForm')) {
            $file = $this->getForm();
        } else {
            $file = JPATH_PLUGINS . '/solidrespayment/' . $this->_name . '/forms/' . $this->_name . '.xml';
        }

        if (empty($data['payments'])
            || empty($file)
            || !is_file($file)
            || !($xml = simplexml_load_file($file))
        ) {
            return;
        }

        $fieldSets = $xml->xpath('fields[@name="payments"]/fieldset');
        $elements  = $xml->xpath('fields[@name="payments"]/fieldset/field');

        if (!isset($fieldSets[0])) {
            return;
        }

        $fieldSet = $fieldSets[0];
        $saveData = [];
        $fields   = [
            $fieldSet['name'] . '_base_rate',
            $fieldSet['name'] . '_base_rate_value',
            $fieldSet['name'] . '_visibility',
        ];

        foreach ($elements as $element) {
            if (isset($element['name'])) {
                $fields[] = (string)$element['name'];
            }
        }

        foreach ($data['payments'] as $k => $v) {
            if (in_array($k, $fields)) {
                if (is_array($v) || is_object($v)) {
                    $registry = new Registry($v);
                    $v        = (string)$registry->toString();
                }

                $saveData[$k] = $v;
            }
        }

        if (is_callable([$this, 'prepareSaveData'])) {
            call_user_func_array([$this, 'prepareSaveData'], [$table, &$saveData]);
        }

        if (!empty($saveData)) {
            $config = $this->getConfig($table->id);
            $config->set($saveData);
        }
    }

    protected function createToken($reservationTable)
    {
        $hashFields = [
            (int)$reservationTable->id,
            trim($reservationTable->code),
            (int)$reservationTable->reservation_asset_id,
            trim($reservationTable->reservation_asset_name),
        ];

        return md5(join(':', $hashFields));
    }

    protected function getCancelPaymentUrl($token)
    {
        $uri = Uri::getInstance();
        $url = Route::_(
            'index.php?option=com_solidres&task=reservation.cancelPayment&token=' . $token . '&identifier=' . $this->identifier,
            false
        );

        return trim($uri->toString(['host', 'scheme']) . $url);
    }

    protected function getReturnPaymentUrl($token)
    {
        $uri = Uri::getInstance();
        $url = Route::_(
            'index.php?option=com_solidres&task=reservation.returnPayment&token=' . $token . '&identifier=' . $this->identifier,
            false
        );

        return trim($uri->toString(['host', 'scheme']) . $url);
    }

    protected function isValidPaymentMethod($identifier)
    {
        if ($identifier != $this->identifier) {
            return false;
        }

        return true;
    }

    protected function log($msg, $priority = Log::DEBUG, $category = '')
    {
        if (empty($category)) {
            $category = $this->identifier;
        }

        Log::add($msg, $priority, $category);
    }

    protected function addPaymentHistory($reservationTable, $title = '', $description = '', $amount = 0)
    {
        try {
            if ($reservationTable instanceof Table) {
                $paymentHistoryData = [
                    'scope'                 => 0,
                    'reservation_id'        => $reservationTable->id,
                    'payment_status'        => $reservationTable->payment_status,
                    'payment_method_id'     => $this->identifier,
                    'payment_amount'        => $amount > 0 ? $amount : ($this->amountToBePaid ?: $reservationTable->total_paid),
                    'payment_method_txn_id' => empty($reservationTable->payment_method_txn_id) ? null : $reservationTable->payment_method_txn_id,
                    'payment_data'          => $reservationTable->payment_data,
                    'title'                 => $title ?: Text::sprintf(
                        'SR_PAY_FOR_RESERVATION_CODE_AT_ASSET_FORMAT',
                        $reservationTable->code,
                        $reservationTable->reservation_asset_name
                    ),
                    'description'           => $description,
                ];
            } elseif (is_array($reservationTable)) {
                $paymentHistoryData = $reservationTable;

                if (!isset($paymentHistoryData['title'])) {
                    $paymentHistoryData['title'] = $title;
                }

                if (!isset($paymentHistoryData['description'])) {
                    $paymentHistoryData['description'] = $description;
                }

                if (empty($paymentHistoryData['payment_method_txn_id'])) {
                    $paymentHistoryData['payment_method_txn_id'] = null;
                }
            } else {
                return false;
            }

            return SolidresHelper::savePaymentHistory($paymentHistoryData, false);
        } catch (RuntimeException $e) {
            $this->log(
                'Payment ' . ucfirst($this->identifier) . ' process payment history fail. Message: ' . $e->getMessage()
            );
        }

        return false;
    }

    public static function hasCardForm($element)
    {
        PluginHelper::importPlugin('solidrespayment', $element);
        $class = 'PlgSolidrespayment' . ucfirst($element);

        return (class_exists($class) && defined($class . '::HAS_CARD_FORM'));
    }
}
