<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Solidres\Reservation;

defined('_JEXEC') or die;

use DateTimeZone;
use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Date\Date;
use Joomla\CMS\Factory;
use Joomla\CMS\Filter\InputFilter;
use Joomla\CMS\Language\Language;
use Joomla\CMS\Language\LanguageFactoryInterface;
use Joomla\CMS\Language\Text;
use Joomla\CMS\Mail\MailerFactoryInterface;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\Component\Solidres\Administrator\Helper\SolidresHelper;
use Joomla\Database\DatabaseInterface;
use Solidres\Config\Config;
use Solidres\Currency\Currency;
use Solidres\Event\ReservationPrepareLinkEvent;
use Solidres\Event\SolidresGenerateQRCodeEvent;
use Solidres\Event\SolidresReservationEmailEvent;
use Solidres\Layout\LayoutHelper;
use Solidres\MVC\FactoryTrait;
use Solidres\Type\CustomFieldHelperInterface;
use Solidres\Utility\Utility;

/**
 * Reservation handler class
 *
 * @package       Solidres
 * @subpackage    Reservation
 */
class Reservation
{
    use FactoryTrait;

    protected DatabaseInterface $dbo;

    public function __construct()
    {
        $this->dbo = Factory::getContainer()->get(DatabaseInterface::class);
    }

    /**
     * Check a room to see if it is allowed to be booked in the period from $checkin -> $checkout
     *
     * @param   int     $roomId
     * @param   string  $checkin
     * @param   string  $checkout
     * @param   int     $bookingType  0 is booking per night and 1 is booking per day
     * @param   int     $excludeId
     * @param   int     $confirmationState
     *
     * @return bool  True if the room is ready to be booked, False otherwise
     */
    public function isRoomAvailable(
        $roomId,
        $checkin,
        $checkout,
        $bookingType = 0,
        $excludeId = 0,
        $confirmationState = 5
    ) {
        $checkinQuote  = $this->dbo->quote($checkin);
        $checkoutQuote = $this->dbo->quote($checkout);

        $query = $this->dbo->getQuery(true);
        $query->select('checkin, checkout');
        $query->from($this->dbo->quoteName('#__sr_reservations') . ' as res');

        $query->join(
            'INNER',
            $this->dbo->quoteName('#__sr_reservation_room_xref') . ' as room
									ON res.id = room.reservation_id
									AND room.room_id = ' . $this->dbo->quote(
                $roomId
            ) . ($excludeId > 0 ? ' AND res.id != ' . (int)$excludeId : '')
        );
        //$query->where('res.checkout >= '. $this->dbo->quote(date('Y-m-d')));
        $query->where(
            "(
									(res.checkin BETWEEN $checkinQuote AND $checkoutQuote OR res.checkout BETWEEN $checkinQuote AND $checkoutQuote)
									OR
									($checkinQuote BETWEEN res.checkin AND res.checkout OR $checkoutQuote BETWEEN res.checkin AND res.checkout)
								)"
        );
        $query->where('res.state = ' . (int)$confirmationState);
        $query->order('res.checkin');

        $this->dbo->setQuery($query);
        $result = $this->dbo->loadObjectList();

        if (is_array($result)) {
            foreach ($result as $currentReservation) {
                $currentCheckin  = $currentReservation->checkin;
                $currentCheckout = $currentReservation->checkout;

                if ($checkin != $checkout) // Normal check
                {
                    if (0 == $bookingType) // Per night
                    {
                        if
                        (
                            ($checkin <= $currentCheckin && $checkout > $currentCheckin) ||
                            ($checkin >= $currentCheckin && $checkout <= $currentCheckout) ||
                            ($checkin < $currentCheckout && $checkout >= $currentCheckout)
                        ) {
                            return false;
                        }
                    } else // Per day
                    {
                        if
                        (
                            ($checkin <= $currentCheckin && $checkout >= $currentCheckin) ||
                            ($checkin >= $currentCheckin && $checkout <= $currentCheckout) ||
                            ($checkin <= $currentCheckout && $checkout >= $currentCheckout)
                        ) {
                            return false;
                        }
                    }
                } else // Edge case when we check for available of a single date (checkin == checkout)
                {
                    if (0 == $bookingType) // Per night
                    {
                        if
                        (
                            ($checkin <= $currentCheckin && $checkout > $currentCheckin) ||
                            ($checkin >= $currentCheckin && $checkout < $currentCheckout) ||
                            ($checkin < $currentCheckout && $checkout >= $currentCheckout)
                        ) {
                            return false;
                        }
                    } else {
                        if
                        (
                            ($checkin <= $currentCheckin && $checkout > $currentCheckin) ||
                            ($checkin >= $currentCheckin && $checkout < $currentCheckout) ||
                            ($checkin <= $currentCheckout && $checkout >= $currentCheckout)
                        ) {
                            return false;
                        }
                    }
                }
            }
        }

        return true;
    }

    /**
     * Check a room to see if it is allowed to be booked in the period from $checkin -> $checkout
     *
     * @param   int     $roomId
     * @param   string  $checkin
     * @param   string  $checkout
     * @param   int     $bookingType  0 is booking per night and 1 is booking per day
     * @param   int     $excludeId
     *
     * @return bool  True if the room is blocked, False otherwise
     *
     * @since 2.1.0
     */
    public function isRoomLimited($roomId, $checkin, $checkout, $bookingType = 0, $excludeId = 0)
    {
        if (!PluginHelper::isEnabled('solidres', 'limitbooking')) {
            return false;
        }

        $query = $this->dbo->getQuery(true);

        $checkinMySQLFormat  = "STR_TO_DATE(" . $this->dbo->quote($checkin) . ", '%Y-%m-%d')";
        $checkoutMySQLFormat = "STR_TO_DATE(" . $this->dbo->quote($checkout) . ", '%Y-%m-%d')";

        if (0 == $bookingType) // Booking per night
        {
            $query->select(
                'COUNT(*) FROM ' . $this->dbo->quoteName('#__sr_limit_booking_details') . '
									WHERE room_id = ' . $this->dbo->quote($roomId) . ' AND 
									limit_booking_id IN (SELECT id FROM ' . $this->dbo->quoteName(
                    '#__sr_limit_bookings'
                ) . '
									WHERE
									(
										(' . $checkinMySQLFormat . ' <= start_date AND ' . $checkoutMySQLFormat . ' > start_date )
										OR
										(' . $checkinMySQLFormat . ' >= start_date AND ' . $checkoutMySQLFormat . ' <= end_date )
										OR
										(start_date != end_date AND ' . $checkinMySQLFormat . ' <= end_date AND ' . $checkoutMySQLFormat . ' >= end_date )
										OR
										(start_date = end_date AND ' . $checkinMySQLFormat . ' <= end_date AND ' . $checkoutMySQLFormat . ' > end_date )
									)
									AND state = 1 ' . ($excludeId > 0 ? ' AND id != ' . (int)$excludeId : '') . '
									)'
            );
        } else // Booking per day
        {
            $query->select(
                'COUNT(*) FROM ' . $this->dbo->quoteName('#__sr_limit_booking_details') . '
									WHERE room_id = ' . $this->dbo->quote($roomId) . ' AND 
									limit_booking_id IN (SELECT id FROM ' . $this->dbo->quoteName(
                    '#__sr_limit_bookings'
                ) . '
									WHERE
									(
										(' . $checkinMySQLFormat . ' <= start_date AND ' . $checkoutMySQLFormat . ' >= start_date )
										OR
										(' . $checkinMySQLFormat . ' >= start_date AND ' . $checkoutMySQLFormat . ' <= end_date )
										OR
										(' . $checkinMySQLFormat . ' <= end_date AND ' . $checkoutMySQLFormat . ' >= end_date )
									)
									AND state = 1 ' . ($excludeId > 0 ? ' AND id != ' . (int)$excludeId : '') . '
									)'
            );
        }

        $this->dbo->setQuery($query);

        if ($this->dbo->loadResult() > 0) {
            return true;
        }

        return false;
    }

    /**
     * Store reservation data and related data like children ages or other room preferences
     *
     * @param   int    $reservationId
     * @param   array  $room  Room information
     *
     * @return void
     */
    public function storeRoom($reservationId, $room)
    {
        $query = $this->dbo->getQuery(true);

        // Store main room info
        $query->insert($this->dbo->quoteName('#__sr_reservation_room_xref'));
        $columns = [
            $this->dbo->quoteName('reservation_id'),
            $this->dbo->quoteName('room_id'),
            $this->dbo->quoteName('room_label'),
            $this->dbo->quoteName('adults_number'),
            $this->dbo->quoteName('children_number'),
            $this->dbo->quoteName('guest_fullname'),
            $this->dbo->quoteName('room_price'),
            $this->dbo->quoteName('room_price_tax_incl'),
            $this->dbo->quoteName('room_price_tax_excl'),
        ];

        if (isset($room['tariff_id']) && !is_null($room['tariff_id'])) {
            $columns = array_merge($columns, [
                $this->dbo->quoteName('tariff_id'),
                $this->dbo->quoteName('tariff_title'),
                $this->dbo->quoteName('tariff_description'),
            ]);
        }

        $query->columns($columns);

        $values = (int)$reservationId . ',' .
            $this->dbo->quote($room['room_id']) . ',' .
            $this->dbo->quote($room['room_label']) . ',' .
            $this->dbo->quote((int)($room['adults_number'] ?? 0)) . ',' .
            $this->dbo->quote((int)($room['children_number'] ?? 0)) . ',' .
            $this->dbo->quote($room['guest_fullname'] ?? '') . ',' .
            $this->dbo->quote($room['room_price'] ?? 0) . ',' .
            $this->dbo->quote($room['room_price_tax_incl'] ?? 0) . ',' .
            $this->dbo->quote($room['room_price_tax_excl'] ?? 0);

        if (isset($room['tariff_id']) && !is_null($room['tariff_id'])) {
            $values .= ',' . $this->dbo->quote($room['tariff_id']) . ',' .
                $this->dbo->quote($room['tariff_title']) . ',' .
                $this->dbo->quote($room['tariff_description']);
        }

        $query->values($values);

        $this->dbo->setQuery($query);
        $this->dbo->execute();

        // Store related data
        $recentInsertedId = $this->dbo->insertid();

        if (isset($room['children_number']) && isset($room['children_ages'])) {
            for ($i = 0; $i < $room['children_number']; $i++) {
                $query->clear();
                $query->insert($this->dbo->quoteName('#__sr_reservation_room_details'));
                $query->columns([
                    $this->dbo->quoteName('reservation_room_id'),
                    $this->dbo->quoteName('key'),
                    $this->dbo->quoteName('value'),
                ]);
                $query->values(
                    (int)$recentInsertedId . ',' .
                    $this->dbo->quote('child' . ($i + 1)) . ',' .
                    $this->dbo->quote($room['children_ages'][$i])
                );

                $this->dbo->setQuery($query);
                $this->dbo->execute();
            }
        }

        if (isset($room['preferences'])) {
            foreach ($room['preferences'] as $key => $value) {
                $query->clear();
                $query->insert($this->dbo->quoteName('#__sr_reservation_room_details'));
                $query->columns([
                    $this->dbo->quoteName('reservation_room_id'),
                    $this->dbo->quoteName('key'),
                    $this->dbo->quoteName('value'),
                ]);
                $query->values(
                    (int)$recentInsertedId . ',' .
                    $this->dbo->quote($key) . ',' .
                    $this->dbo->quote($value)
                );

                $this->dbo->setQuery($query);
                $this->dbo->execute();
            }
        }
    }

    /**
     * Store extra information
     *
     * @param   int     $reservationId
     * @param   int     $roomId
     * @param   string  $roomLabel
     * @param   int     $extraId
     * @param   string  $extraName
     * @param   int     $extraQuantity  The extra quantity or NULL if extra does not have quantity
     * @param   int     $price
     *
     * @return void
     */
    public function storeExtra(
        $reservationId,
        $roomId,
        $roomLabel,
        $extraId,
        $extraName,
        $extraQuantity = null,
        $price = 0
    ) {
        $query = $this->dbo->getQuery(true);
        $query->insert($this->dbo->quoteName('#__sr_reservation_room_extra_xref'));
        $query->columns([
            $this->dbo->quoteName('reservation_id'),
            $this->dbo->quoteName('room_id'),
            $this->dbo->quoteName('room_label'),
            $this->dbo->quoteName('extra_id'),
            $this->dbo->quoteName('extra_name'),
            $this->dbo->quoteName('extra_quantity'),
            $this->dbo->quoteName('extra_price'),
        ]);
        $query->values(
            $this->dbo->quote($reservationId) . ',' .
            $this->dbo->quote($roomId) . ',' .
            $this->dbo->quote($roomLabel) . ',' .
            $this->dbo->quote($extraId) . ',' .
            $this->dbo->quote($extraName) . ',' .
            ($extraQuantity === null ? null : $this->dbo->quote($extraQuantity)) . ',' .
            $this->dbo->quote($price)
        );
        $this->dbo->setQuery($query);
        $this->dbo->execute();
    }

    /**
     * Check for the validity of check in and check out date
     *
     * Conditions
     * + Number of days a booking must be made in advance
     * + Maximum length of stay
     *
     * @param   string  $checkIn
     * @param   string  $checkOut
     * @param   array   $conditions
     *
     * @return Boolean
     * @throws Exception
     */
    public function isCheckInCheckOutValid($checkIn, $checkOut, $conditions)
    {
        $offset   = Factory::getApplication()->get('offset');
        $checkIn  = new Date($checkIn, $offset);
        $checkOut = new Date($checkOut, $offset);
        $today    = new Date('now', $offset);
        $today->setTime(0, 0, 0);

        if ($checkIn < $today || $checkOut < $today) {
            throw new Exception('SR_ERROR_PAST_CHECKIN_CHECKOUT', 50005);
        }

        if ((!isset($conditions['booking_type']) && (isset($conditions['min_length_of_stay']) && $conditions['min_length_of_stay'] > 0))
            ||
            (isset($conditions['booking_type']) && $conditions['booking_type'] == 0)
        ) {
            if ($checkOut <= $checkIn) {
                throw new Exception('SR_ERROR_INVALID_CHECKIN_CHECKOUT', 50001);
            }
        }

        // Interval between check in and check out date
        if (isset($conditions['min_length_of_stay']) && $conditions['min_length_of_stay'] > 0) {
            $interval1 = $checkOut->diff($checkIn)->format('%a');

            if ($interval1 < $conditions['min_length_of_stay']) // count nights, not days
            {
                throw new Exception('SR_ERROR_INVALID_MIN_LENGTH_OF_STAY_' . $conditions['booking_type'], 50002);
            }
        }

        // Interval between checkin and today
        $interval2 = $checkIn->diff($today)->format('%a');

        if (isset($conditions['min_days_book_in_advance']) && $conditions['min_days_book_in_advance'] > 0) {
            if ($interval2 < $conditions['min_days_book_in_advance']) {
                throw new Exception('SR_ERROR_INVALID_MIN_DAYS_BOOK_IN_ADVANCE', 50003);
            }
        }

        if (isset($conditions['max_days_book_in_advance']) && $conditions['max_days_book_in_advance'] > 0) {
            if ($interval2 > $conditions['max_days_book_in_advance']) {
                throw new Exception('SR_ERROR_INVALID_MAX_DAYS_BOOK_IN_ADVANCE', 50004);
            }
        }

        return true;
    }

    /**
     * Send email
     *
     * @param   int     $reservationId  The reservation to get the reservation info for emails (Optional)
     * @param   string  $state          The target reservation state that trigger sending email
     * @param   int     $emailType      0 is the reservation completion email, 1 is for booking requires approval, 2 is for
     *                                  status changing email
     *
     * @return bool True if email sending completed successfully. False otherwise
     * @since  0.1.0
     *
     */
    public function sendEmail($reservationId = null, $state = '', $emailType = 0)
    {
        $subject                      = [];
        $body                         = [];
        $app                          = Factory::getApplication();
        $solidresConfig               = ComponentHelper::getParams('com_solidres');
        $config                       = $app->getConfig();
        $cmsActiveLang                = $app->getLanguage();
        $cmsDefaultLangTag            = ComponentHelper::getParams('com_languages')->get('site', 'en-GB');
        $cmsDefaultLang               = Language::getInstance($cmsDefaultLangTag, $config->get('debug_lang'));
        $dateFormat                   = $solidresConfig->get('date_format', 'd-m-Y');
        $cancellationState            = $solidresConfig->get('cancel_state', 4);
        $tzoffset                     = $config->get('offset');
        $timezone                     = new DateTimeZone($tzoffset);
        $context                      = 'com_solidres.reservation.process';
        $savedReservationId           = $reservationId ?? $app->getUserState($context . '.savedReservationId');
        $modelReservation             = $this->createModel('Reservation');
        $modelProperty                = $this->createModel('Reservationasset');
        $reservation                  = $modelReservation->getItem($savedReservationId);
        $stayLength                   = (int)Utility::calculateDateDiff(
            $reservation->checkin,
            $reservation->checkout
        );
        $direction                    = $app->getDocument()->direction;
        $customerLanguage             = !empty($reservation->customer_language) ? $reservation->customer_language : null;
        $customerEmail                = $reservation->customer_email;
        $languageOverridden           = false;
        $isCancelled                  = $state == $cancellationState;
        $reservationStatusesList      = SolidresHelper::getStatusesList(0);
        $reservationStatuses          = [];
        $reservationStatusesEmailText = [];
        $inputFilter                  = InputFilter::getInstance();

        $modelProperty->setState('property_info_only', true);
        $property         = $modelProperty->getItem($reservation->reservation_asset_id);
        $enableTouristTax = $property->params['enable_tourist_tax'] ?? false;
        $emailAttachments = $property->params['email_attachments'] ?? [];

        foreach ($reservationStatusesList as $status) {
            $reservationStatuses[$status->value]          = $status->text;
            $reservationStatusesEmailText[$status->value] = $status->email_text;
        }

        $cmsActiveLang->load('com_solidres', JPATH_ADMINISTRATOR . '/components/com_solidres', null, true);
        $cmsActiveLang->load('com_solidres', JPATH_SITE . '/components/com_solidres', null, true);

        // Load override language file
        $cmsActiveLang->load(
            'com_solidres_category_' . $property->category_id,
            JPATH_BASE . '/components/com_solidres',
            null,
            true
        );

        // Override CMS language if the customer selected language is different
        if ($customerLanguage
            && ($customerLanguage !== $cmsDefaultLangTag || $customerLanguage !== $cmsActiveLang->getTag())) {
            $customerLang = Factory::getContainer()->get(LanguageFactoryInterface::class)->createLanguage(
                $customerLanguage
            );

            foreach ($cmsActiveLang->getPaths() as $extension => $files) {
                if (strpos($extension, 'plg_system') !== false) {
                    $extension_name = substr($extension, 11);

                    $customerLang->load($extension, JPATH_ADMINISTRATOR)
                    || $customerLang->load($extension, JPATH_PLUGINS . '/system/' . $extension_name);

                    $cmsDefaultLang->load($extension, JPATH_ADMINISTRATOR)
                    || $cmsDefaultLang->load($extension, JPATH_PLUGINS . '/system/' . $extension_name);

                    continue;
                }

                foreach ($files as $file => $loaded) {
                    $customerLang->load(
                        $extension,
                        preg_replace('#/language/' . $cmsActiveLang->getTag() . '/.*$#', '', $file)
                    );
                    $cmsDefaultLang->load(
                        $extension,
                        preg_replace('#/language/' . $cmsActiveLang->getTag() . '/.*$#', '', $file)
                    );
                }
            }

            Factory::$language = $customerLang;
            $app->loadLanguage($customerLang);
            $languageOverridden = true;
        }

        $bankWireInstructions = [];

        if ($reservation->payment_method_id == 'bankwire') {
            $solidresPaymentConfigData               = new Config(['scope_id' => $reservation->reservation_asset_id]);
            $bankWireInstructions['account_name']    = Utility::translateText(
                $solidresPaymentConfigData->get('payments/bankwire/bankwire_accountname')
            );
            $bankWireInstructions['account_details'] = Utility::translateText(
                $solidresPaymentConfigData->get('payments/bankwire/bankwire_accountdetails')
            );
        }

        // Prepare some costs data to be showed
        $baseCurrency = new Currency(0, $reservation->currency_id);
        $costs        = Utility::prepareReservationCosts($reservation);

        extract($costs);

        $commissions = $reservation->commissions ?? [];

        $baseCost = 0;
        foreach ($commissions as $commission) {
            if (0 == $commission->rate_type) {
                $baseCost = $commission->revenues - $commission->commissions;
            }
        }

        $displayData = [
            'reservation'                     => $reservation,
            'subTotal'                        => $subTotal->format(),
            'totalDiscount'                   => $reservation->total_discount > 0.00 ? $totalDiscount->format() : null,
            'tax'                             => $tax->format(),
            'touristTax'                      => $touristTax->format(),
            'totalFee'                        => $totalFee->format(),
            'totalExtraPriceTaxExcl'          => $totalExtraPriceTaxExcl->format(),
            'totalExtraTax'                   => $totalExtraTax->format(),
            'grandTotal'                      => $grandTotal->format(),
            'stayLength'                      => $stayLength,
            'depositAmount'                   => $depositAmount->format(),
            'totalPaid'                       => $totalPaid->format(),
            'totalDue'                        => $totalDue->format(),
            'bankwireInstructions'            => $bankWireInstructions,
            'asset'                           => $property,
            'dateFormat'                      => $dateFormat,
            'timezone'                        => $timezone,
            'baseCurrency'                    => $baseCurrency,
            'paymentMethodLabel'              => 'SR_PAYMENT_METHOD_' . $reservation->payment_method_id,
            'paymentMethodCustomEmailContent' => $app->getUserState(
                $context . '.payment_method_custom_email_content',
                ''
            ),
            'discountPreTax'                  => $reservation->discount_pre_tax,
            'direction'                       => $direction,
            'enableTouristTax'                => $enableTouristTax,
            'paymentMethodSurcharge'          => $paymentMethodSurcharge->format(),
            'paymentMethodDiscount'           => $paymentMethodDiscount->format(),
            'customerFields'                  => [],
            'roomFields'                      => [],
            'qrCode'                          => null,
            'baseCost'                        => $baseCost,
        ];

        if (PluginHelper::isEnabled('solidres', 'customfield')) {
            /** @var $fieldHelper CustomFieldHelperInterface */
            $fieldHelper = \Solidres\Factory::get(CustomFieldHelperInterface::class);
            $cid         = [(int)$property->category_id];
            $fields      = $fieldHelper::findFields(['context' => 'com_solidres.customer'], $cid);

            $fieldHelper::setFieldDataValues(
                $fieldHelper::getValues(['context' => 'com_solidres.customer.' . $reservation->id])
            );
            $renderValue = function ($field) use ($reservation, $fieldHelper) {
                $value = $fieldHelper::displayFieldValue($field->field_name, null, true);

                if ($field->type == 'file') {
                    $code     = $reservation->code;
                    $email    = strip_tags($fieldHelper::displayFieldValue('customer_email', null, true));
                    $fileName = basename($value);

                    if (strpos($fileName, '_') !== false) {
                        $parts    = explode('_', $fileName, 2);
                        $fileName = $parts[1];
                    }

                    $value = '<a href="' . Uri::root() . 'index.php?option=com_solidres&task=option=com_solidres&task=customfield.downloadFile&file=' . base64_encode(
                            strip_tags($value)
                        ) . '&email=' . $email . '&code=' . $code . '" target="_blank">' . $fileName . '</a>';
                }

                return $value;
            };

            foreach ($fields as $field) {
                if ($field->field_name != 'customer_email2') {
                    $displayData['customerFields'][] = [
                        'title' => Text::_($field->title),
                        'value' => trim($renderValue($field)),
                    ];
                }
            }

            $roomFields = $fieldHelper::findFields(['context' => 'com_solidres.room']);

            if (!empty($roomFields)) {
                foreach ($reservation->reserved_room_details as $room) {
                    $roomFieldsValues = $fieldHelper::getValues(['context' => 'com_solidres.room.' . $room->id]);
                    $fieldHelper::setFieldDataValues($roomFieldsValues);

                    foreach ($roomFields as $roomField) {
                        $displayData['roomFields'][$room->id][$roomField->id] = [
                            'title'   => $roomField->title,
                            'value'   => $fieldHelper::displayFieldValue($roomField, null, true),
                            'attribs' => $roomField->attribs,
                        ];
                    }
                }
            }

            $displayData['customerNote'] = $fieldHelper::displayFieldValue('customer_note');
        } else {
            $fields = [
                'SR_CUSTOMER_TITLE'        => 'customer_title',
                'SR_FIRSTNAME'             => 'customer_firstname',
                'SR_MIDDLENAME'            => 'customer_middlename',
                'SR_LASTNAME'              => 'customer_lastname',
                'JGLOBAL_EMAIL'            => 'customer_email',
                'SR_PHONE'                 => 'customer_phonenumber',
                'SR_MOBILEPHONE'           => 'customer_mobilephone',
                'SR_COMPANY'               => 'customer_company',
                'SR_ADDRESS_1'             => 'customer_address1',
                'SR_ADDRESS_2'             => 'customer_address2',
                'SR_CITY'                  => 'customer_city',
                'SR_ZIP'                   => 'customer_zipcode',
                'SR_FIELD_COUNTRY_LABEL'   => 'customer_country_name',
                'SR_FIELD_GEO_STATE_LABEL' => 'customer_geostate_name',
                'SR_VAT_NUMBER'            => 'customer_vat_number',
                'SR_NOTES'                 => 'note',
            ];

            foreach ($fields as $string => $fieldName) {
                $displayData['customerFields'][] = [
                    'title' => Text::_($string),
                    'value' => trim($reservation->{$fieldName}),
                ];
            }

            $displayData['customerNote'] = $reservation->note;
        }

        // Prepare email subject and greeting text for customer, taking customer selected language into consideration
        switch ($emailType) {
            case 0:
                $subject[$customerEmail] = Text::_('SR_EMAIL_RESERVATION_COMPLETE');

                if (!empty($reservationStatusesEmailText[$state])) {
                    $displayData['greetingText'] = $this->prepareReservationStatusChangeEmailBody(
                        $property,
                        $reservation
                    );
                } else {
                    $displayData['greetingText'] = ['SR_EMAIL_GREETING_TEXT', $property->name];
                }

                break;
            case 1:
                $subject[$customerEmail]     = Text::_('SR_EMAIL_BOOKING_APPROVAL_NOTICE');
                $displayData['greetingText'] = ['SR_EMAIL_GREETING_TEXT_APPROVAL', $property->name];
                break;
            case 2:
                if ($isCancelled) {
                    $subject[$customerEmail]     = Text::_('SR_EMAIL_RESERVATION_CANCELLED');
                    $displayData['greetingText'] = [
                        'SR_EMAIL_GREETING_TEXT_CANCELLED',
                        $reservation->code,
                        $property->name,
                    ];
                } else {
                    $subject[$customerEmail]     = Text::_('SR_RESERVATION_STATUS_CHANGE_EMAIL_SUBJECT');
                    $displayData['greetingText'] = $this->prepareReservationStatusChangeEmailBody(
                        $property,
                        $reservation
                    );
                }
                break;
        }

        $displayData['forceLang'] = $customerLanguage;

        $body[$customerEmail] = LayoutHelper::render(
            'emails.reservation_complete_customer_html_inliner',
            $displayData
        );

        // Send email to customer
        $mail = Factory::getContainer()->get(MailerFactoryInterface::class)->createMailer();
        $mail->setSender([$config->get('mailfrom'), $config->get('fromname')]);
        $mail->addReplyTo($property->email);
        $mail->addRecipient($customerEmail);
        $mail->setSubject($subject[$customerEmail]);
        $mail->setBody($body[$customerEmail]);
        $mail->isHtml(true);

        foreach ($emailAttachments as $emailAttachment) {
            $mail->addAttachment(
                $inputFilter->clean($emailAttachment['attachment_path'], 'STRING'),
            );
        }

        if (PluginHelper::isEnabled('solidres', 'invoice')) {
            $invoiceTable = $this->createTable('Invoice');
            $invoiceTable->load(['reservation_id' => $savedReservationId]);
            LayoutHelper::addIncludePath(JPATH_PLUGINS . '/solidres/invoice/layouts');

            if (PluginHelper::isEnabled('solidres', 'qrcode')) {
                $qrCodeData = json_encode([
                    'id'         => $reservation->id,
                    'code'       => $reservation->code,
                    'grandTotal' => $displayData['grandTotal'],
                ]);

                PluginHelper::importPlugin('solidres', 'qrcode');
                $displayData['qrCode'] = SolidresGenerateQRCodeEvent::dispatch(
                    [
                        'data' => $qrCodeData,
                        'code' => $displayData['qrCode'],
                    ]
                )->getCode();
            }

            $pdf = LayoutHelper::render(
                'emails.reservation_complete_customer_pdf',
                $displayData,
                false
            );

            if ($solidresConfig->get('enable_pdf_attachment', 1) == 1) {
                $this->getPDFAttachment($mail, $pdf, $savedReservationId, $reservation->code);
            }

            $selectedPaymentMethod    = $app->getUserState($context . '.payment_method_id', '');
            $autoSendPaymentMethods   = $solidresConfig->get('auto_send_invoice_payment_methods', '');
            $sendInvoiceAutomatically = ($solidresConfig->get('auto_create_invoice', 0) == 1 &&
                $solidresConfig->get('auto_send_invoice', 0) == 1
                && in_array($selectedPaymentMethod, $autoSendPaymentMethods));

            if ($sendInvoiceAutomatically) {
                PluginHelper::importPlugin('solidres');
                $invoiceFolder      = JPATH_ROOT . '/media/com_solidres/invoices/';
                $attachmentFileName = $solidresConfig->get('solidres_invoice_pdf_file_name', 'Invoice');

                if ($invoiceTable->id) {
                    $mail->addAttachment(
                        $invoiceFolder . $invoiceTable->filename,
                        $attachmentFileName . '_' . $invoiceTable->invoice_number . '.pdf',
                        'base64',
                        'application/pdf'
                    );
                }
            }
        }

        if (!$mail->send()) {
            if ($languageOverridden) {
                // Revert CMS language
                Factory::$language = $cmsDefaultLang;
                $app->loadLanguage($cmsDefaultLang);
            }

            return false;
        } else {
            if (!empty($sendInvoiceAutomatically)
                && isset($invoiceTable)
            ) {
                $invoiceTable->set('sent_on', Factory::getDate()->toSql());
                $invoiceTable->store();
            }
        }

        if ($languageOverridden) {
            // Revert CMS language
            Factory::$language = $cmsDefaultLang;
            $app->loadLanguage($cmsDefaultLang);
        }

        // Send to the property owners and partners, use the default language
        $editLinks = [
            'admin'   => Uri::root() . 'administrator/index.php?option=com_solidres&view=reservation&layout=edit&id=' . $reservation->id,
            'partner' => Uri::root() . 'index.php?option=com_solidres&task=reservationform.edit&id=' . $reservation->id,
        ];

        if (PluginHelper::isEnabled('solidres', 'app')) {
            PluginHelper::importPlugin('solidres', 'app');
            $editLinks = ReservationPrepareLinkEvent::dispatch(['subject' => $reservation, 'editLinks' => $editLinks])
                ->getEditLinks();
        }

        $displayData['editLink'] = $editLinks['admin'];

        if (isset($partner) && !empty($partner->email) && $partner->email == $property->email) {
            $displayData['editLink'] = $editLinks['partner'];
        }

        $displayData['forceLang'] = null;

        switch ($emailType) {
            case 0:
                $subject[$property->email] = Text::sprintf(
                    'SR_EMAIL_NEW_RESERVATION_NOTIFICATION',
                    $reservation->code,
                    $reservation->customer_firstname,
                    $reservation->customer_lastname
                );

                $displayData['greetingText'] = ['SR_EMAIL_NOTIFICATION_GREETING_TEXT', $displayData['editLink']];
                break;
            case 1:
                $subject[$property->email]   = Text::sprintf(
                    'SR_FIELD_BOOKING_INQUIRY_NOTIFICATION_OWNER_EMAIL_SUBJECT',
                    $reservation->code,
                    $reservation->customer_firstname,
                    $reservation->customer_lastname
                );
                $displayData['greetingText'] = ['SR_EMAIL_GREETING_TEXT_APPROVAL_OWNER', $displayData['editLink']];
                break;
            case 2:

                if ($isCancelled) {
                    $subject[$property->email]   = Text::sprintf(
                        'SR_EMAIL_RESERVATION_CANCELLED_NOTIFICATION',
                        $reservation->code,
                        $reservation->customer_firstname,
                        $reservation->customer_lastname
                    );
                    $displayData['greetingText'] = [
                        'SR_EMAIL_NOTIFICATION_GREETING_TEXT_CANCELLED',
                        $reservation->code,
                        $displayData['editLink'],
                    ];
                } else {
                    $subject[$property->email]   = Text::_('SR_RESERVATION_STATUS_CHANGE_EMAIL_SUBJECT');
                    $displayData['greetingText'] = $this->prepareReservationStatusChangeEmailBody(
                        $property,
                        $reservation,
                        1
                    );
                }
                break;
        }

        if (PluginHelper::isEnabled('solidres', 'hub') && !empty($property->partner_id)) {
            $modelCustomer = $this->createModel('Customer');
            $partner       = $modelCustomer->getItem($property->partner_id);

            if (!empty($partner->email)
                && $partner->email != $property->email
            ) {
                $subject['partner'] = $subject[$property->email];
            }
        }

        $body[$property->email] = LayoutHelper::render(
            'emails.reservation_complete_owner_html_inliner',
            $displayData
        );

        $mail2 = Factory::getContainer()->get(MailerFactoryInterface::class)->createMailer();
        $mail2->setSender([$config->get('mailfrom'), $config->get('fromname')]);
        $mail2->addRecipient($property->email);
        $additionalNotificationEmails = [];
        if (isset($property->params['additional_notification_emails']) && !empty($property->params['additional_notification_emails'])) {
            $additionalNotificationEmails = explode(',', $property->params['additional_notification_emails']);
        }

        if (!empty($additionalNotificationEmails)) {
            $mail2->addRecipient($additionalNotificationEmails);
        }

        $mail2->setSubject($subject[$property->email]);
        $mail2->setBody($body[$property->email]);
        $mail2->isHtml(true);

        if (!$mail2->send()) {
            return false;
        }

        // Send to the property partner
        if (isset($partner)
            && !empty($partner->email)
            && isset($subject['partner'])
        ) {
            switch ($emailType) {
                case 0:
                    $displayData['greetingText'] = ['SR_EMAIL_NOTIFICATION_GREETING_TEXT', $editLinks['partner']];
                    break;
                case 1:
                    $displayData['greetingText'] = ['SR_EMAIL_GREETING_TEXT_APPROVAL_OWNER', $editLinks['partner']];
                    break;
                case 2:

                    if ($state == $cancellationState) {
                        $displayData['greetingText'] = [
                            'SR_EMAIL_NOTIFICATION_GREETING_TEXT_CANCELLED',
                            $reservation->code,
                            $editLinks['partner'],
                        ];
                    } else {
                        $displayData['greetingText'] = $this->prepareReservationStatusChangeEmailBody(
                            $property,
                            $reservation,
                            1
                        );
                    }

                    break;
            }

            $displayData['editLink'] = $editLinks['partner'];
            $body['partner']         = LayoutHelper::render(
                'emails.reservation_complete_owner_html_inliner',
                $displayData
            );

            $recipients = [
                $partner->email,
            ];

            $db    = $this->dbo;
            $query = $db->getQuery(true)
                ->select('u.email')
                ->from($db->quoteName('#__users', 'u'))
                ->join('INNER', $db->quoteName('#__sr_property_staff_xref', 'a') . ' ON a.staff_id = u.id')
                ->where('u.block = 0 AND a.property_id = ' . (int)$property->id);
            $db->setQuery($query);

            if ($staffEmails = $db->loadColumn()) {
                $recipients = array_unique(array_merge($recipients, $staffEmails));
            }

            $mail3 = Factory::getContainer()->get(MailerFactoryInterface::class)->createMailer();
            $mail3->setSender([$config->get('mailfrom'), $config->get('fromname')]);
            $mail3->addRecipient($recipients);
            $mail3->setSubject($subject['partner']);
            $mail3->setBody($body['partner']);
            $mail3->isHtml(true);

            if (!$mail3->send()) {
                return false;
            }
        }

        return true;
    }

    /**
     * Prepare the email body to be sent when reservation status changed
     *
     * @param $property
     * @param $reservation
     * @param $recipientType  0 is the customer, 1 is the owner/partner
     *
     * @return array|string|string[]
     *
     * @since version
     */
    protected function prepareReservationStatusChangeEmailBody($property, $reservation, $recipientType = 0)
    {
        $emailFormat                  = ComponentHelper::getParams('com_solidres')->get(
            'reservation_state_change_email_format',
            0
        );
        $reservationStatusesList      = SolidresHelper::getStatusesList(0);
        $reservationStatuses          = [];
        $reservationStatusesEmailText = [];

        foreach ($reservationStatusesList as $status) {
            $reservationStatuses[$status->value]          = $status->text;
            $reservationStatusesEmailText[$status->value] = $status->email_text;
        }

        // If this status has a custom email text, use it instead of the default one
        if (!empty($reservationStatusesEmailText[$reservation->state])) {
            $adultsNumber   = 0;
            $childrenNumber = 0;
            foreach ($reservation->reserved_room_details as $detail) {
                $adultsNumber   += $detail->adults_number;
                $childrenNumber += $detail->children_number;
            }

            $isDiscountPreTax       = $reservation->discount_pre_tax;
            $totalExtraPriceTaxIncl = $reservation->total_extra_price_tax_incl;
            $baseCurrency           = new Currency(0, $reservation->currency_id);

            if ($isDiscountPreTax) {
                $grandTotal = $reservation->total_price_tax_excl - $reservation->total_discount + $reservation->tax_amount + $totalExtraPriceTaxIncl;
            } else {
                $grandTotal = $reservation->total_price_tax_excl + $reservation->tax_amount - $reservation->total_discount + $totalExtraPriceTaxIncl;
            }

            $grandTotal += $reservation->tourist_tax_amount;
            $grandTotal += $reservation->total_fee;
            $baseCurrency->setValue($grandTotal);
            $grandTotalCurrency = $baseCurrency->format();
            $baseCurrency->setValue($reservation->deposit_amount);
            $depositAmountCurrency = $baseCurrency->format();

            $commissions = $reservation->commissions ?? [];

            $baseCost = 0;
            foreach ($commissions as $commission) {
                if (0 == $commission->rate_type) {
                    $baseCost = $commission->revenues - $commission->commissions;
                }
            }

            $body = str_replace(
                [
                    '{property_name}',
                    '{customer_title}',
                    '{customer_firstname}',
                    '{customer_lastname}',
                    '{customer_email}',
                    '{customer_phone}',
                    '{customer_mobilephone}',
                    '{res_code}',
                    '{res_status}',
                    '{checkin}',
                    '{checkout}',
                    '{guest_number}',
                    '{adults_number}',
                    '{children_number}',
                    '{grand_total}',
                    '{deposit_amount}',
                    '{base_cost}',
                ],
                [
                    ($recipientType == 1 && !empty($property->alternative_name) ? $property->alternative_name : $property->name),
                    $reservation->customer_title,
                    $reservation->customer_firstname,
                    $reservation->customer_lastname,
                    $reservation->customer_email,
                    $reservation->customer_phonenumber,
                    $reservation->customer_mobilephone,
                    $reservation->code,
                    $reservationStatuses[$reservation->state],
                    $reservation->checkin,
                    $reservation->checkout,
                    ($adultsNumber + $childrenNumber),
                    $adultsNumber,
                    $childrenNumber,
                    $grandTotalCurrency,
                    $depositAmountCurrency,
                    $baseCost,
                ],
                $reservationStatusesEmailText[$reservation->state]
            );
        } else {
            if (1 == $emailFormat) {
                $body = Text::sprintf(
                    'SR_RESERVATION_STATUS_CHANGE_EMAIL_CONTENT_HTML',
                    $reservation->code,
                    $reservationStatuses[$reservation->state],
                    $property->name
                );
            } else {
                $body = Text::sprintf(
                    'SR_RESERVATION_STATUS_CHANGE_EMAIL_CONTENT',
                    ($reservation->customer_firstname . ' ' . $reservation->customer_lastname),
                    $reservation->code,
                    $reservationStatuses[$reservation->state],
                    $property->name
                );
            }
        }

        return $body;
    }

    /**
     * Generate unique string for Reservation
     *
     * @param   string  $srcString  The string that need to be calculate checksum
     *
     * @return string The unique string for each Reservation
     */
    public function getCode($srcString)
    {
        return hash('crc32', $srcString . uniqid());
    }

    /**
     * Create PDF attachment.
     *
     * @param $mail        mail object.
     * @param $reid        reservation id.
     * @param $reCode      reservation code.
     *
     * @since 1.0.0
     */
    protected function getPDFAttachment($mail, $content, $reid, $reCode)
    {
        PluginHelper::importPlugin('solidres');
        $solidresConfig     = ComponentHelper::getParams('com_solidres');
        $attachmentFileName = $solidresConfig->get('solidres_voucher_pdf_file_name', 'voucher');
        $results            = SolidresReservationEmailEvent::dispatch(['content' => $content, 'reservationId' => $reid])
            ->getResult();

        if ($results) {
            $mail->addAttachment(
                $results[0],
                $attachmentFileName . '_' . $reCode . '.pdf',
                'base64',
                'application/pdf'
            );
        }
    }

    public function sendGenericReservationStatusChangeEmail($reservationId = null)
    {
        $app                          = Factory::getApplication();
        $modelReservation             = $this->createModel('Reservation');
        $modelAsset                   = $this->createModel('ReservationAsset');
        $reservation                  = $modelReservation->getItem($reservationId);
        $asset                        = $modelAsset->getItem($reservation->reservation_asset_id);
        $recipients                   = [$asset->email];
        $config                       = $app->getConfig();
        $additionalNotificationEmails = !empty($asset->params['additional_notification_emails']) ? explode(
            ',',
            $asset->params['additional_notification_emails']
        ) : [];
        $cmsActiveLang                = $app->getLanguage();
        $cmsActiveLangTag             = $cmsActiveLang->getTag();
        $cmsDefaultLangTag            = ComponentHelper::getParams('com_languages')->get('site', 'en-GB');
        $cmsDefaultLang               = Factory::getContainer()
            ->get(LanguageFactoryInterface::class)
            ->createLanguage($cmsDefaultLangTag, $config->get('debug_lang'));

        $cmsActiveLang->load('com_solidres', JPATH_ADMINISTRATOR . '/components/com_solidres', null, true);
        $cmsActiveLang->load('com_solidres', JPATH_SITE . '/components/com_solidres', null, true);

        if (!empty($additionalNotificationEmails)) {
            foreach ($additionalNotificationEmails as $additionalNotificationEmail) {
                if (!in_array($additionalNotificationEmail, $recipients)) {
                    $recipients[] = $additionalNotificationEmail;
                }
            }
        }

        if ($result = $this->sendGenericReservationStatusChangeToRecipients($recipients, $reservation, $asset)) {
            $customerLanguage = $reservation->customer_language;
            $recipients       = [$reservation->customer_email];
            $overrideCmsLang  = $customerLanguage && ($customerLanguage !== $cmsDefaultLangTag || $customerLanguage !== $cmsActiveLangTag);

            if ($overrideCmsLang) {
                $customerLang = Factory::getContainer()->get(LanguageFactoryInterface::class)->createLanguage(
                    $customerLanguage
                );

                foreach ($cmsActiveLang->getPaths() as $extension => $files) {
                    if (strpos($extension, 'plg_system') !== false) {
                        $extension_name = substr($extension, 11);

                        $customerLang->load($extension, JPATH_ADMINISTRATOR)
                        || $customerLang->load($extension, JPATH_PLUGINS . '/system/' . $extension_name);

                        $cmsDefaultLang->load($extension, JPATH_ADMINISTRATOR)
                        || $cmsDefaultLang->load($extension, JPATH_PLUGINS . '/system/' . $extension_name);

                        continue;
                    }

                    foreach ($files as $file => $loaded) {
                        $customerLang->load(
                            $extension,
                            preg_replace('#/language/' . $cmsActiveLangTag . '/.*$#', '', $file)
                        );
                        $cmsDefaultLang->load(
                            $extension,
                            preg_replace('#/language/' . $cmsActiveLangTag . '/.*$#', '', $file)
                        );
                    }
                }

                // Override CMS language
                Factory::$language = $customerLang;
                $app->loadLanguage($customerLang);
            }

            $result = $this->sendGenericReservationStatusChangeToRecipients($recipients, $reservation, $asset);

            if ($overrideCmsLang) {
                // Revert CMS language
                Factory::$language = $cmsDefaultLang;
                $app->loadLanguage($cmsDefaultLang);
            }
        }

        return $result;
    }

    protected function sendGenericReservationStatusChangeToRecipients($recipients, $reservation, $property)
    {
        $body = $this->prepareReservationStatusChangeEmailBody($property, $reservation);

        $config = Factory::getApplication()->getConfig();
        $mail   = Factory::getContainer()->get(MailerFactoryInterface::class)->createMailer();
        $mail->setSender([$config->get('mailfrom'), $config->get('fromname')]);
        $mail->addRecipient($recipients);
        $mail->setSubject(Text::_('SR_RESERVATION_STATUS_CHANGE_EMAIL_SUBJECT'));
        $mail->setBody($body);
        $mail->isHtml(false);

        return $mail->send();
    }

    public function hasCheckIn($roomTypeID, $checkin)
    {
        $dbo   = Factory::getContainer()->get(DatabaseInterface::class);
        $query = $dbo->getQuery(true);
        $query->select('COUNT(*)')->from('#__sr_reservations AS a')
            ->innerJoin('#__sr_rooms AS b ON b.room_type_id = ' . (int)$roomTypeID)
            ->innerJoin('#__sr_reservation_room_xref AS c ON c.room_id = b.id AND c.reservation_id = a.id')
            ->where('a.checkin = ' . $dbo->quote($checkin) . ' AND a.state != -2');

        return $dbo->setQuery($query)->loadResult() > 0;
    }

    public function hasLimitBookingStartDate($roomTypeID, $date)
    {
        if (!PluginHelper::isEnabled('solidres', 'limitbooking')) {
            return false;
        }

        $query = $this->dbo->getQuery(true);
        $query->select('COUNT(*)')->from($this->dbo->quoteName('#__sr_limit_bookings', 'a'))
            ->innerJoin($this->dbo->quoteName('#__sr_rooms', 'b') . ' ON b.room_type_id = ' . (int)$roomTypeID)
            ->innerJoin(
                $this->dbo->quoteName(
                    '#__sr_limit_booking_details',
                    'c'
                ) . ' ON c.room_id = b.id AND c.limit_booking_id = a.id'
            )
            ->where('a.start_date = ' . $this->dbo->quote($date) . ' AND a.state = 1');

        return $this->dbo->setQuery($query)->loadResult() > 0;
    }
}
