<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Joomla\Plugin\System\Solidres\Extension;

defined('_JEXEC') or die;

use DateInterval;
use JLoader;
use Joomla\CMS\Event\Model\AfterSaveEvent;
use Joomla\CMS\Event\Model\BeforeSaveEvent;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Event\SubscriberInterface;
use Joomla\Filesystem\Folder;
use Joomla\Registry\Registry;
use Solidres\ClassAlias;
use Solidres\Config\Config;
use Solidres\Event\SolidresAfterInitialiseDocumentEvent;
use Solidres\Event\SolidresPluginRegisterEvent;
use Solidres\Media\ImageUploaderHelper;
use Solidres\Plugin\Plugin;
use Solidres\Utility\Utility;

JLoader::registerNamespace('Solidres', JPATH_LIBRARIES . '/solidres/src');

// B/C register the class aliases, this will be removed in the next stabled version
ClassAlias::register();

if (file_exists(JPATH_LIBRARIES . '/solidres/defines.php')) {
    require_once JPATH_LIBRARIES . '/solidres/defines.php';
}

class Solidres extends Plugin implements SubscriberInterface
{
    protected static string $prevMediaUploadPathBase = '';

    public static function getSubscribedEvents(): array
    {
        return [
            'onAfterRoute'              => 'onAfterRoute',
            'onAfterInitialiseDocument' => 'onAfterInitialiseDocument',
            'onAfterRender'             => 'onAfterRender',
            'onExtensionBeforeSave'     => 'onExtensionBeforeSave',
            'onExtensionAfterSave'      => 'onExtensionAfterSave',
        ];
    }

    public function onAfterRoute()
    {
        $app = $this->getApplication();

        if ($app->input->get('option') == 'com_solidres') {
            // Load core language files to be used in customer dashboard and partner dashboard
            $language = $app->getLanguage();
            $language->load('', JPATH_ADMINISTRATOR);
            $language->load('', JPATH_ROOT);
            $language->load('com_solidres', JPATH_ADMINISTRATOR . '/components/com_solidres', null, true);
        }

        PluginHelper::importPlugin('user');
        PluginHelper::importPlugin('solidres');
        SolidresPluginRegisterEvent::dispatch();

        if (PluginHelper::isEnabled('solidres', 'advancedextra')) {
            JLoader::register('SRExtra', JPATH_PLUGINS . '/solidres/advancedextra/libraries/extra/extra.php');
        }

        if (!defined('SR_LAYOUT_STYLE')) {
            define('SR_LAYOUT_STYLE', $this->solidresConfig->get('layout_style', ''));
        }

        // B/C for ACL config
        if (PluginHelper::isEnabled('solidres', 'acl')
            && $app->input->get('option') == 'com_config'
            && $app->input->get('component') == 'com_solidres'
            && is_dir(JPATH_PLUGINS . '/solidres/acl/administrator/components/com_solidres/models/forms')
        ) {
            Form::addFormPath(JPATH_PLUGINS . '/solidres/acl/administrator/components/com_solidres/models/forms');
        }
    }

    public function onAfterInitialiseDocument()
    {
        $dateFormat = $this->solidresConfig->get('date_format', 'd-m-Y');
        $app        = $this->getApplication();
        $app->getDocument()->addScriptOptions('com_solidres.general', [
            'JVersion'                => SR_ISJ5 ? 5 : (SR_ISJ4 ? 4 : 3),
            'ChannelManager'          => (PluginHelper::isEnabled('solidres', 'channelmanager') ? 1 : 0),
            'AutoScroll'              => (int)$this->solidresConfig->get('enable_auto_scroll', 1),
            'AutoScrollTariff'        => (int)$this->solidresConfig->get('auto_scroll_tariff', 1),
            'EnableUnoccupiedPricing' => (int)$this->solidresConfig->get('enable_unoccupied_pricing', 0),
            'MinLengthOfStay'         => (int)$this->solidresConfig->get('min_length_of_stay', 1),
            'MinDaysBookInAdvance'    => (int)$this->solidresConfig->get('min_days_book_in_advance', 0),
            'MaxDaysBookInAdvance'    => (int)$this->solidresConfig->get('max_days_book_in_advance', 0),
            'DatePickerMonthNum'      => (int)$this->solidresConfig->get('datepicker_month_number', 2),
            'WeekStartDay'            => (int)$this->solidresConfig->get('week_start_day', 1),
            'DateFormat'              => $dateFormat,
            'DateFormatJS'            => Utility::convertDateFormatPattern($dateFormat),
            'GoogleMapsAPIKey'        => $this->solidresConfig->get('google_map_api_key', ''),
        ]);

        SolidresAfterInitialiseDocumentEvent::dispatch();
    }

    public function onAfterRender()
    {
        $app                = $this->getApplication();
        $solidresConfigData = new Config(['scope_id' => 0, 'data_namespace' => 'system']);
        $lastUpdateCheck    = $solidresConfigData->get('system/last_update_check', '');
        $needUpdateChecking = false;
        $updateSourceFile   = JPATH_ADMINISTRATOR . '/components/com_solidres/views/system/cache/updates.json';

        if ($app->isClient('administrator')) {
            if (empty($lastUpdateCheck) || !is_file($updateSourceFile)) {
                $needUpdateChecking = true;
            } else {
                $now     = Factory::getDate('now', 'UTC');
                $nextRun = Factory::getDate($lastUpdateCheck, 'UTC');
                $nextRun->add(new DateInterval('PT24H'));

                if ($now->toUnix() > $nextRun->toUnix()) {
                    $needUpdateChecking = true;
                }
            }

            if ($needUpdateChecking) {
                $solidresSystemCtrl = $this->getSolidresMVCFactory()->createController(
                    'System',
                    'Administrator',
                    [],
                    $app,
                    $app->getInput()
                );
                $url                = 'https://www.solidres.com/checkupdates';
                $solidresSystemCtrl->postFindUpdates($url);
                $solidresConfigData->set(['last_update_check' => Factory::getDate('now', 'UTC')->toUnix()]);
            }
        }

        if ($this->solidresConfig->get('enable_multilingual_mode', 1) == 1) {
            if ($app->isClient('administrator')) {
                return true;
            }

            $buffer = $app->getBody();

            if (strpos($buffer, '{lang') === false) {
                return true;
            }

            $regexTextarea = "#<textarea(.*?)>(.*?)<\/textarea>#is";
            $regexInput    = "#<input(.*?)>#is";

            $matches = [];
            preg_match_all($regexTextarea, $buffer, $matches, PREG_SET_ORDER);
            $textarea = [];
            foreach ($matches as $key => $match) {
                if (strpos($match[0], '{lang') !== false) {
                    $textarea[$key] = $match[0];
                    $buffer         = str_replace($textarea[$key], '~^t' . $key . '~', $buffer);
                }
            }

            $matches = [];
            preg_match_all($regexInput, $buffer, $matches, PREG_SET_ORDER);
            $input = [];
            foreach ($matches as $key => $match) {
                if (
                    (strpos($match[0], 'type="password"') !== false ||
                        strpos($match[0], 'type="text"') !== false) &&
                    strpos($match[0], '{lang') !== false
                ) {
                    $input[$key] = $match[0];
                    $buffer      = str_replace($input[$key], '~^i' . $key . '~', $buffer);
                }
            }

            if (strpos($buffer, '{lang') !== false) {
                $buffer = Utility::filterText($buffer);

                if ($textarea) {
                    foreach ($textarea as $key => $t) {
                        $buffer = str_replace('~^t' . $key . '~', $t, $buffer);
                    }
                    unset($textarea);
                }

                if ($input) {
                    foreach ($input as $key => $i) {
                        $buffer = str_replace('~^i' . $key . '~', $i, $buffer);
                    }
                    unset($input);
                }

                $app->setBody($buffer);
            }

            unset($buffer);
        }
    }

    public function onExtensionBeforeSave(BeforeSaveEvent $event)
    {
        $context = $event->getArgument('context');

        if ($context === 'com_config.component' && $this->getApplication()->input->getCmd(
                'component'
            ) === 'com_solidres') {
            static::$prevMediaUploadPathBase = ImageUploaderHelper::getUploadPath(true);
        }
    }

    public function onExtensionAfterSave(AfterSaveEvent $event)
    {
        $context = $event->getArgument('context');
        $table   = $event->getArgument('subject');

        if ($context !== 'com_config.component' || $this->getApplication()->input->getCmd(
                'component'
            ) !== 'com_solidres') {
            return;
        }

        $params      = new Registry($table->params);
        $newPathBase = $params->get('images_storage_path');
        $currentPath = JPATH_ROOT . '/images/' . static::$prevMediaUploadPathBase;
        $newPath     = JPATH_ROOT . '/images/' . $newPathBase;

        if (
            static::$prevMediaUploadPathBase === $newPathBase
            || !is_dir($currentPath)
            || !($folders = Folder::folders($currentPath, '[p|r|e]', false, true))
        ) {
            return;
        }

        if (!is_dir($newPath)) {
            Folder::create($newPath);
        }

        foreach ($folders as $folder) {
            if (Folder::copy($folder, $newPath . '/' . basename($folder), '', true)) {
                Folder::delete($folder);
            }
        }

        if (!Folder::folders($currentPath)) {
            Folder::delete($currentPath);
        }
    }
}
