<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Joomla\Plugin\User\Solidres\Extension;

defined('_JEXEC') or die;

/**
 * Solidres User plugin
 *
 * @package     Solidres
 * @subpackage  Customer
 * @since       0.6.0
 */

use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Event\Model\BeforeSaveEvent;
use Joomla\CMS\Event\Model\PrepareFormEvent;
use Joomla\CMS\Event\User\AfterDeleteEvent;
use Joomla\CMS\Event\User\AfterSaveEvent;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Uri\Uri;
use Joomla\CMS\User\UserFactoryInterface;
use Joomla\CMS\User\UserHelper;
use Joomla\Component\Solidres\Site\Service\UserRouter;
use Joomla\Event\SubscriberInterface;
use Joomla\Registry\Registry;
use Joomla\Utilities\ArrayHelper;
use Solidres\Event\SolidresAttachRulesEvent;
use Solidres\Plugin\Plugin;
use Solidres\Type\CustomFieldHelperInterface;

class Solidres extends Plugin implements SubscriberInterface
{
    public static function getSubscribedEvents(): array
    {
        return [
            'onUserAfterDelete'         => 'onUserAfterDelete',
            'onUserAfterSave'           => 'onUserAfterSave',
            'onCustomerBeforeSave'      => 'onCustomerBeforeSave',
            'onAfterInitialiseDocument' => 'onAfterInitialiseDocument',
            'onSolidresAttachRules'     => 'onSolidresAttachRules',
            'onContentPrepareForm'      => 'contentPrepareForm',
        ];
    }

    public function onUserAfterDelete(AfterDeleteEvent $event)
    {
        $success = $event->getDeletingResult();
        $user    = $event->getUser();

        if (!$success) {
            return false;
        }

        $db            = $this->getDatabase();
        $customerTable = $this->createTable('Customer');
        $customerTable->load(['user_id' => $user['id']]);

        // Handle relationship with Solidres's Customers
        $query = $db->getQuery(true);

        // Take care of Reservation
        $query->update($db->quoteName('#__sr_reservations'))
            ->set('customer_id = NULL')
            ->where('customer_id = ' . $db->quote($customerTable->id));
        $db->setQuery($query)->execute();

        // Take care of Customer Fields
        $query->clear();
        $query->delete()->from($db->quoteName('#__sr_customer_fields'))
            ->where('user_id = ' . $db->quote($customerTable->id));
        $db->setQuery($query)->execute();

        // Take care of relationship with Reservation Asset
        $query->clear();
        $query->update($db->quoteName('#__sr_reservation_assets'))
            ->set('partner_id = NULL')
            ->where('partner_id = ' . $db->quote($customerTable->id));
        $db->setQuery($query)->execute();

        // Take care of Customer itself
        $query->clear();
        $query->delete()->from($db->quoteName('#__sr_customers'))
            ->where('id = ' . $db->quote($customerTable->id));
        $db->setQuery($query)->execute();

        return true;
    }

    public function onUserAfterSave(AfterSaveEvent $event)
    {
        $this->saveUserProfile($event->getUser(), $event->getIsNew(), $event->getSavingResult());
    }

    /*
     * Create a new Joomla user before we create a new Solidres's customer.
     *
     * The procedure is different between front end and back end.
     *
     * @param $data
     * @param $table
     * @param $isNew
     * @param $response
     *
     * @return bool
     */

    protected function saveUserProfile($data, $isNew, $result)
    {
        if (!PluginHelper::isEnabled('solidres', 'customfield')) {
            return false;
        }

        $arrayData = (array)$data;
        $userId    = ArrayHelper::getValue($arrayData, 'id', 0, 'int');

        if ($result) {
            if (empty($arrayData['Solidres_fields'])) {
                $jform                        = $this->getApplication()->input->get('jform', [], 'array');
                $arrayData['Solidres_fields'] = $jform['Solidres_fields'] ?? [];
            }

            $customerTable = $this->createTable('Customer', 'Administrator');

            if (!$customerTable->load(['user_id' => $userId])) {
                $customerTable->set('user_id', $userId);
                $customerTable->set('customer_group_id', null);
                $customerTable->set('customer_code', '');
            }

            /** @var $fieldHelper CustomFieldHelperInterface */
            $fieldHelper = \Solidres\Factory::get(CustomFieldHelperInterface::class);
            $fields      = $fieldHelper::findFields(['context' => 'com_solidres.customer']);

            if ($fields && !empty($arrayData['Solidres_fields'])) {
                $dataValue = [];

                foreach ($fields as $field) {
                    if (isset($arrayData['Solidres_fields'][$field->field_name])) {
                        $value       = $arrayData['Solidres_fields'][$field->field_name];
                        $dataValue[] = [
                            'id'      => 0,
                            'context' => 'com_solidres.customer.profile.' . $userId,
                            'value'   => $value,
                            'storage' => $field,
                        ];

                        if (strpos($field->field_name, 'customer_') === 0) {
                            $name = str_replace('customer_', '', $field->field_name);

                            if (property_exists($customerTable, $name)) {
                                if (empty($value) && in_array($name, ['country_id', 'geo_state_id'])) {
                                    $value = null;
                                }

                                $customerTable->set($name, $value);
                            }
                        }
                    }
                }

                if (count($dataValue)) {
                    $fieldHelper::storeValues($dataValue, $isNew);
                }
            }

            $customerTable->store(true);
        }

        return true;
    }

    public function onCustomerBeforeSave(BeforeSaveEvent $event)
    {
        $table              = $event->getItem();
        $data               = $event->getData();
        $isNew              = $event->getIsNew();
        $app                = $this->getApplication();
        $isSite             = $app->isClient('site');
        $solidresConfig     = ComponentHelper::getParams('com_solidres');
        $customerUserGroups = $solidresConfig->get('customer_user_groups', [2]);
        $lang               = $app->getLanguage();

        $lang->load('com_users', JPATH_SITE);
        $userData = [
            'id'        => $data['user_id'],
            'username'  => $data['username'],
            'password'  => $data['password'],
            'password2' => $data['password2'],
            'password1' => $data['password2'],
            'email'     => $data['email'],
            'email1'    => $data['email'],
        ];

        if (isset($data['Solidres_fields']['customer_firstname'])) {
            $userData['name'] = $data['Solidres_fields']['customer_firstname'];

            if (isset($data['Solidres_fields']['customer_middlename'])) {
                $userData['name'] .= ' ' . $data['Solidres_fields']['customer_middlename'];
            }

            $userData['name'] .= ' ' . $data['Solidres_fields']['customer_lastname'];
        } else {
            $userData['name'] = $data['firstname'] . ' ' . $data['middlename'] . ' ' . $data['lastname'];
        }

        if (!$isSite || !$isNew) // Special case for Customer Dashboard profile editing
        {
            $pk         = (!empty($userData['id'])) ? $userData['id'] : 0;
            $joomlaUser = Factory::getContainer()->get(UserFactoryInterface::class)->loadUserById($pk);

            if (empty($joomlaUser->groups)) {
                $userData['groups'] = $customerUserGroups;
            }

            if (!$joomlaUser->bind($userData)) {
                throw new Exception($joomlaUser->getError(), 500);
            }

            $result = $joomlaUser->save();

            if (!$result) {
                throw new Exception($joomlaUser->getError(), 500);
            }

            // Assign the recent insert joomla user id
            $table->load(['user_id' => $joomlaUser->id]);

            return true;
        }
        // For front end, just use the way Joomla register a user
        Form::addFormPath(JPATH_SITE . '/components/com_users/forms');
        $userFactory = $app->bootComponent('com_users')->getMVCFactory();
        $model       = $userFactory->createModel('Registration', 'Site', ['ignore_request' => true]);

        // Attempt to save the data.
        $return = $model->register($userData);

        // Assign the recent insert joomla user id to response, so that we can store it into customer's table
        if (is_numeric($return)) {
            $table->load(['user_id' => $return]);
        } elseif (in_array($return, ['useractivate', 'adminactivate'])) {
            $table->load(['user_id' => UserHelper::getUserId($userData['username'])]);
        }
    }

    public function contentPrepareForm(PrepareFormEvent $event)
    {
        $form          = $event->getForm();
        $data          = $event->getData();
        $formName      = $form->getName();
        $edit          = $this->getApplication()->getInput()->getWord('layout') === 'edit';
        $allowContexts = [
            'com_solidres.customer',
            'com_admin.profile',
            'com_users.user',
            'com_users.profile',
            'com_users.registration',
        ];

        if (!PluginHelper::isEnabled('solidres', 'customfield') || !in_array($formName, $allowContexts)) {
            return true;
        }

        $language = $this->getApplication()->getLanguage();
        $language->load('com_solidres', JPATH_BASE . '/components/com_solidres')
        || $language->load('com_solidres', JPATH_BASE);
        $id = (int)$form->getValue('id', null, 0);

        // Skip adding our custom fields if user is not consented to the privacy
        if (
            in_array($formName, ['com_users.profile', 'com_users.registration'])
            && !$this->isUserConsented($id > 0 ? $id : $this->getApplication()->getIdentity()->id)
            && PluginHelper::isEnabled('system', 'privacyconsent')
        ) {
            return true;
        }

        if ($id < 1) {
            $registry = new Registry($data);
            $id       = (int)$registry->get('id', 0);
        }

        $ignoreFieldsNames = [
            'customer_note',
            'customer_email',
            'customer_email2',
        ];

        /** @var $fieldHelper CustomFieldHelperInterface */
        $fieldHelper = \Solidres\Factory::get(CustomFieldHelperInterface::class);

        $source = [];
        $fields = $fieldHelper::findFields(['context' => 'com_solidres.customer']);
        $app    = $this->getApplication();

        foreach ($fields as $field) {
            if ($field->type != 'file') {
                $source[] = $field;
            }

            if ($app->isClient('administrator')) {
                $field->optional = 1;
            }
        }

        $xml = $fieldHelper::buildFields($source, 'Solidres_fields', $ignoreFieldsNames);

        if ($form->load($xml->saveXML())) {
            foreach ($form->getGroup('Solidres_fields') as $field) {
                if ($field->getAttribute('name') == 'file') {
                    $form->removeField($field->getAttribute('name'), 'Solidres_fields');
                }
            }

            if ($id > 0) {
                $customerTable = $this->getSolidresMVCFactory()->createTable('Customer', 'Administrator');
                $fieldsData    = [
                    'Solidres_fields' => [],
                ];

                if ($formName == 'com_solidres.customer') {
                    $load = $customerTable->load($id);

                    foreach ($form->getFieldset('fields') as $field) {
                        $form->removeField($field->getAttribute('name'), 'Solidres_fields');
                    }
                } else {
                    $load = $customerTable->load(['user_id' => $id]);
                }

                if ($fieldsValues = $fieldHelper::getValues(
                    ['context' => 'com_solidres.customer.profile.' . $customerTable->user_id]
                )) {
                    foreach ($fieldsValues as $fieldsValue) {
                        if ($name = $fieldsValue->field->get('field_name')) {
                            if ($edit) {
                                $fieldsData['Solidres_fields'][$name] = isset($fieldsValue->orgValue) ? $fieldsValue->orgValue : $fieldsValue->value;
                            } else {
                                $fieldsData['Solidres_fields'][$name] = $fieldsValue->value;
                            }
                        }
                    }
                }

                if ($load) {
                    foreach ($customerTable->getProperties() as $name => $value) {
                        if (strpos($name, 'customer') !== 0) {
                            $name = 'customer_' . $name;
                        }

                        if (!isset($fieldsData['Solidres_fields'][$name])) {
                            $fieldsData['Solidres_fields'][$name] = $value;
                        }
                    }
                }

                $form->bind($fieldsData);
            }
        }

        $this->getApplication()->getDocument()->addScriptDeclaration(
            '
			Solidres.jQuery(document).ready(function($){
				var country = $("#jform_Solidres_fields_customer_country_id");
	
				if (country.length) {
					var state = $("#jform_Solidres_fields_customer_geo_state_id");
					var selected = state.val();
	
					country.on("change", function(){
						var countryId = $(this).val();
	
						if (countryId == "") {
							state.html("").trigger("liszt:updated");
						} else {
							$.ajax({
								url : "' . Uri::root(true) . '/index.php?option=com_solidres&format=json&task=states.find&id=" + countryId,
								type: "post",
								success : function(html) {
									html = $.trim(html);
	
									if (html.indexOf("<option") === 0) {
										state.html(html).val(selected).trigger("liszt:updated");
									} else {
										state.html("").trigger("liszt:updated");
									}
								}
							});
						}
	
					});
	
					country.trigger("change");
				}
			});
		'
        );

        return true;
    }

    /**
     * Method to check if the given user has consented yet
     *
     * @param   int  $userId  ID of uer to check
     *
     * @return  bool
     *
     * @since   3.9.0
     */
    private function isUserConsented($userId)
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true);
        $query->select('COUNT(*)')
            ->from('#__privacy_consents')
            ->where('user_id = ' . (int)$userId)
            ->where('subject = ' . $db->quote('PLG_SYSTEM_PRIVACYCONSENT_SUBJECT'))
            ->where('state = 1');
        $db->setQuery($query);

        return (int)$db->loadResult() > 0;
    }

    public function onAfterInitialiseDocument()
    {
        /** @var Joomla\CMS\WebAsset\WebAssetManager $wa */
        $wa = $this->getApplication()->getDocument()->getWebAssetManager();
        $wa->getRegistry()->addExtensionRegistryFile('com_solidres');

        if ($this->getApplication()->isClient('administrator')) {
            $wa->useScript('com_solidres.admin');
        }

        $wa->useScript('com_solidres.common');
    }

    public function onSolidresAttachRules(SolidresAttachRulesEvent $event)
    {
        return new UserRouter($event->getRouter());
    }

    protected function getMenuTypeOptions()
    {
        return [
            'customer' => $this->siteMVCPath . '/tmpl/customer/default.xml',
        ];
    }
}
