<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Joomla\Plugin\Extension\Solidres\Extension;

defined('_JEXEC') or die;

use Joomla\CMS\Date\Date;
use Joomla\CMS\Event\Model\PrepareFormEvent;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\CMS\Router\Route;
use Joomla\CMS\User\CurrentUserInterface;
use Joomla\CMS\User\CurrentUserTrait;
use Joomla\Component\Solidres\Site\Helper\WishListHelper;
use Joomla\Database\DatabaseAwareInterface;
use Joomla\Database\DatabaseAwareTrait;
use Joomla\Event\DispatcherInterface;
use Joomla\Event\SubscriberInterface;
use RuntimeException;
use Solidres\Event\ReservationChangeStateEvent;
use Solidres\Plugin\Plugin;
use Solidres\Reservation\Reservation;

/**
 * Solidres Extension Plugin
 *
 * @package        Solidres
 * @subpackage     Extension.Plugin
 * @since          1.5
 */
class Solidres extends Plugin implements DatabaseAwareInterface, CurrentUserInterface, SubscriberInterface
{
    use DatabaseAwareTrait;
    use CurrentUserTrait;

    private static array $roomTypeFieldsCache = [];
    protected $autoloadLanguage = true;
    private int $confirmationState;
    private string $context;

    public function __construct(DispatcherInterface $dispatcher, array $config = [])
    {
        parent::__construct($dispatcher, $config);

        $this->confirmationState = $this->solidresConfig->get('confirm_state', 5);
        $this->context           = 'com_solidres.reservation.process';
    }

    public static function getSubscribedEvents(): array
    {
        return [
            'onReservationAssetPrepareForm' => 'reservationAssetPrepareForm',
            'onRoomTypePrepareForm'         => 'roomTypePrepareForm',
            'onReservationChangeState'      => 'onReservationChangeState',
            'onAjaxSolidres'                => 'onAjaxSolidres',
        ];
    }

    /**
     * @param   Form   $form  The form to be altered.
     * @param   array  $data  The associated data for the form.
     *
     * @return    bool
     * @since    1.6
     */
    public function reservationAssetPrepareForm(PrepareFormEvent $event)
    {
        $form = $event->getForm();
        // Check we are manipulating a valid form.
        if (!in_array($form->getName(), ['com_solidres.reservationasset'])) {
            return true;
        }

        // Add the registration fields to the form.
        Form::addFormPath(JPATH_PLUGINS . '/extension/solidres/fields');
        $form->loadFile('reservationasset', false);

        // Toggle whether the checkin time field is required.
        if ($this->params->get('param_reservation_asset_checkin_time', 1) > 0) {
            $form->setFieldAttribute(
                'checkin_time',
                'required',
                $this->params->get('param_reservation_asset_checkin_time') == 2,
                'reservationasset_extra_fields'
            );
        } else {
            $form->removeField('checkin_time', 'reservationasset_extra_fields');
        }

        // Toggle whether the checkout time field is required.
        if ($this->params->get('param_reservation_asset_checkout_time', 1) > 0) {
            $form->setFieldAttribute(
                'checkout_time',
                'required',
                $this->params->get('param_reservation_asset_checkout_time') == 2,
                'reservationasset_extra_fields'
            );
        } else {
            $form->removeField('checkout_time', 'reservationasset_extra_fields');
        }

        // Toggle whether the cancellation prepayment field is required.
        if ($this->params->get('param_reservation_asset_cancellation_prepayment', 1) > 0) {
            $form->setFieldAttribute(
                'cancellation_prepayment',
                'required',
                $this->params->get('param_reservation_asset_cancellation_prepayment') == 2,
                'reservationasset_extra_fields'
            );
        } else {
            $form->removeField('cancellation_prepayment', 'reservationasset_extra_fields');
        }

        // Toggle whether the children and extra beds field is required.
        if ($this->params->get('param_reservation_asset_children_and_extra_beds', 1) > 0) {
            $form->setFieldAttribute(
                'children_and_extra_beds',
                'required',
                $this->params->get('param_reservation_asset_children_and_extra_beds') == 2,
                'reservationasset_extra_fields'
            );
        } else {
            $form->removeField('children_and_extra_beds', 'reservationasset_extra_fields');
        }

        // Toggle whether the children and extra beds field is required.
        if ($this->params->get('param_reservation_asset_pets', 1) > 0) {
            $form->setFieldAttribute(
                'pets',
                'required',
                $this->params->get('param_reservation_asset_pets') == 2,
                'reservationasset_extra_fields'
            );
        } else {
            $form->removeField('pets', 'reservationasset_extra_fields');
        }

        // Toggle whether the facebook field is required.
        if ($this->params->get('param_reservation_asset_facebook', 1) > 0) {
            $form->setFieldAttribute(
                'facebook',
                'required',
                $this->params->get('param_reservation_asset_facebook') == 2,
                'reservationasset_extra_fields'
            );
        } else {
            $form->removeField('facebook', 'reservationasset_extra_fields');
        }

        // Toggle whether the twitter field is required.
        if ($this->params->get('param_reservation_asset_twitter', 1) > 0) {
            $form->setFieldAttribute(
                'twitter',
                'required',
                $this->params->get('param_reservation_asset_twitter') == 2,
                'reservationasset_extra_fields'
            );
        } else {
            $form->removeField('twitter', 'reservationasset_extra_fields');
        }


        return true;
    }


    /**
     * @param   Form   $form  The form to be altered.
     * @param   array  $data  The associated data for the form.
     *
     * @return    bool
     * @since    1.6
     */
    public function roomTypePrepareForm(PrepareFormEvent $event)
    {
        $form = $event->getForm();

        // Check we are manipulating a valid form.
        if (!in_array($form->getName(), ['com_solidres.roomtype'])) {
            return true;
        }

        // Add the registration fields to the form.
        Form::addFormPath(JPATH_PLUGINS . '/extension/solidres/fields');
        $form->loadFile('roomtype', false);

        // Toggle whether the checkin time field is required.
        if ($this->params->get('param_roomtype_room_facilities', 1) > 0) {
            $form->setFieldAttribute(
                'room_facilities',
                'required',
                $this->params->get('param_roomtype_room_facilities') == 2,
                'roomtype_custom_fields'
            );
        } else {
            $form->removeField('room_facilities', 'roomtype_custom_fields');
        }

        // Toggle whether the checkout time field is required.
        if ($this->params->get('param_roomtype_room_size', 1) > 0) {
            $form->setFieldAttribute(
                'room_size',
                'required',
                $this->params->get('param_roomtype_room_size') == 2,
                'roomtype_custom_fields'
            );
        } else {
            $form->removeField('room_size', 'roomtype_custom_fields');
        }

        // Toggle whether the cancellation prepayment field is required.
        if ($this->params->get('param_roomtype_bed_size', 1) > 0) {
            $form->setFieldAttribute(
                'bed_size',
                'required',
                $this->params->get('param_roomtype_bed_size') == 2,
                'roomtype_custom_fields'
            );
        } else {
            $form->removeField('bed_size', 'roomtype_custom_fields');
        }

        return true;
    }

    /**
     * Hook into reservation state changing event and send outgoing emails
     *
     * @param $context
     * @param $pks
     * @param $value
     * @param $oldValue
     *
     * @return mixed
     *
     * @since 1.8.0
     */
    public function onReservationChangeState(ReservationChangeStateEvent $event)
    {
        $context = $event->getContext();
        $value   = $event->getNewState();
        $pks     = $event->getIdList();

        if ($context != 'com_solidres.changestate' || $this->solidresConfig->get(
                'send_email_on_reservation_state_change',
                1
            ) == 0) {
            return true;
        }

        $acceptedStatuses = $this->solidresConfig->get('status_trigger_reservation_state_change_email', []);

        if (!in_array($value, $acceptedStatuses)) {
            return true;
        }

        $reservationTable = $this->createTable('Reservation');

        if ($value == $this->confirmationState) {
            $reservationTable->load($pks[0]);
            $reservationTable->set('confirmation_date', Date::getInstance()->format('Y-m-d h:i:s'));
            $reservationTable->store();
        }

        $solidresReservation = new Reservation();
        $cancellationState   = $this->solidresConfig->get('cancel_state', 4);
        $emailFormat         = $this->solidresConfig->get('reservation_state_change_email_format', 0);

        // If the reservation's status is changed from any statuses to "Cancelled", send a normal HTML reservation email
        // Since v2.13.0, we have a new option to send status change email in HTML format as well
        if ($value == $cancellationState || 1 == $emailFormat) {
            return $solidresReservation->sendEmail($pks[0], $value, 2);
        } else // For other status changes, send a text email to notify user
        {
            return $solidresReservation->sendGenericReservationStatusChangeEmail($pks[0]);
        }
    }


    public function onAjaxSolidres()
    {
        $app      = Factory::getApplication();
        $objectId = $app->input->getUint('objectId');
        $itemId   = $app->input->getUint('Itemid');
        $scope    = $app->input->getString('scope');
        $wishList = WishListHelper::getInstance($scope);

        if ($scope == 'experience' && !PluginHelper::isEnabled('solidres', 'experience')) {
            throw new RuntimeException('The Solidres experience is not enabled.');
        }

        if (strtolower($app->input->getString('type')) == 'add') {
            if ($scope == 'experience') {
                $objectTable = $this->createTable('Experience');
            } else {
                $objectTable = $this->createTable('ReservationAsset');
            }

            if ($objectTable->load($objectId)) {
                $history             = ['id' => $objectTable->id, 'name' => $objectTable->name];
                $item                = $wishList->add($objectId, $history)->load($objectId);
                $item['history']     = $history;
                $item['wishListUrl'] = Route::_(
                    "index.php?option=com_solidres&view=wishlist&scope=$scope&Itemid=$itemId",
                    false
                );

                return json_encode($item);
            } else {
                throw new RuntimeException('Object ID ' . $objectId . ' not found. Scope: ' . $scope);
            }
        }

        $wishList->clear($objectId);
        $list = $wishList->load($objectId);

        return empty($list) ? 0 : count($list);
    }
}
