<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Joomla\Component\Solidres\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Solidres\MVC\Model\AdminModel;
use Solidres\Coupon\Coupon;
use Solidres\Currency\Currency;
use Solidres\Media\ImageUploaderTrait;
use Solidres\Utility\Utility;
use Throwable;

class ExtraModel extends AdminModel
{
    use ImageUploaderTrait;

    public function save($data)
    {
        if ($result = parent::save($data)) {
            $dbo   = $this->getDatabase();
            $query = $dbo->getQuery(true);
            $pk    = (int) $this->getState('extra.id');

            $query->delete($dbo->quoteName('#__sr_extra_coupon_xref'))
                ->where('extra_id = ' . $dbo->quote($pk));
            $dbo->setQuery($query)
                ->execute();

            if (!empty($data['coupon_id'])) {
                foreach ($data['coupon_id'] as $value) {
                    $query->clear();
                    $query->insert($dbo->quoteName('#__sr_extra_coupon_xref'))
                        ->columns([$dbo->quoteName('extra_id'), $dbo->quoteName('coupon_id')])
                        ->values($pk . ',' . (int)$value);
                    $dbo->setQuery($query)->execute();
                }
            }

            try {
                $this->uploadMedia($pk);
            } catch (Throwable $e) {
            }
        }

        return $result;
    }

    public function getTable($name = 'Extra', $prefix = 'Administrator', $options = [])
    {
        return parent::getTable($name, $prefix, $options);
    }

    protected function prepareTable($table)
    {
        $table->name = htmlspecialchars_decode($table->name, ENT_QUOTES);

        if (empty($table->id)) {
            // Set ordering to the last item if not set
            if (empty($table->ordering)) {
                $db    = $this->getDatabase();
                $query = $db->getQuery(true);
                $query->clear();
                $query->select('MAX(ordering)')->from($db->quoteName('#__sr_extras'));
                $db->setQuery($query);
                $max = $db->loadResult();

                $table->ordering = $max + 1;
            }
        }

        // If tax_id is empty, then set it to null
        if ($table->tax_id === '') {
            $table->tax_id = null;
        }

        // If tax_id is empty, then set it to nulll
        if ($table->coupon_id === '') {
            $table->coupon_id = null;
        }
    }

    protected function canDelete($record)
    {
        $app  = Factory::getApplication();
        $user = $app->getIdentity();

        if ($app->isClient('api')) {
            // Authorization has already checked from the ApiController
            return true;
        }

        if ($app->isClient('administrator')) {
            return parent::canDelete($record);
        }

        return Utility::isAssetPartner($user->id, $record->reservation_asset_id);
    }

    protected function canEditState($record)
    {
        $app  = Factory::getApplication();
        $user = $app->getIdentity();

        if ($app->isClient('api')) {
            // Authorization has already checked from the ApiController
            return true;
        }

        if ($app->isClient('administrator')) {
            return parent::canEditState($record);
        } else {
            return Utility::isAssetPartner($user->get('id'), $record->reservation_asset_id);
        }
    }

    public function getItem($pk = null)
    {
        $item = parent::getItem($pk);

        if (isset($item->id)) {
            $app              = Factory::getApplication();
            $mvcFactory       = $this->getMVCFactory();
            $solidresConfig   = ComponentHelper::getParams('com_solidres');
            $numberOfDecimals = $solidresConfig->get('number_decimal_points', 2);
            $showTaxIncl      = $this->getState($this->getName() . '.show_price_with_tax', 0);
            $assetTable       = $mvcFactory->createTable('Reservationasset', 'Administrator');
            $taxTable         = $mvcFactory->createTable('Tax', 'Administrator');
            $assetTable->load($item->reservation_asset_id);
            $solidresCurrency = new Currency(0, $assetTable->currency_id);
            $taxTable->load($item->tax_id);
            $taxAmount      = 0;
            $taxAdultAmount = 0;
            $taxChildAmount = 0;
            $coupon         = $app->getUserState('com_solidres.reservation.process.coupon');
            $solidresCoupon = new Coupon();

            $item->price       = $itemPrice = round($item->price, $numberOfDecimals);
            $item->price_adult = $itemPriceAdult = round($item->price_adult, $numberOfDecimals);
            $item->price_child = $itemPriceChild = round($item->price_child, $numberOfDecimals);

            // Coupon calculation if available
            if (isset($coupon) && is_array($coupon) && $app->isClient('site')) {
                $isCouponApplicable = $solidresCoupon->isApplicable($coupon['coupon_id'], $item->id, 'extra');
                if ($isCouponApplicable) {
                    if ($coupon['coupon_is_percent'] == 1) {
                        $deductionAmountP = $item->price * ($coupon['coupon_amount'] / 100);
                        $deductionAmountA = $item->price_adult * ($coupon['coupon_amount'] / 100);
                        $deductionAmountC = $item->price_child * ($coupon['coupon_amount'] / 100);
                    } else {
                        $deductionAmountP = $deductionAmountA = $deductionAmountC = $coupon['coupon_amount'];
                    }
                    $item->price       -= $deductionAmountP;
                    $item->price_adult -= $deductionAmountA;
                    $item->price_child -= $deductionAmountC;
                }
            }

            if (!empty($taxTable->rate)) {
                if (isset($item->price_includes_tax) && $item->price_includes_tax == 1) {
                    $taxAmount      = $item->price - ($item->price / (1 + $taxTable->rate));
                    $taxAdultAmount = $item->price_adult - ($item->price_adult / (1 + $taxTable->rate));
                    $taxChildAmount = $item->price_child - ($item->price_child / (1 + $taxTable->rate));

                    $itemPrice      -= $taxAmount;
                    $itemPriceAdult -= $taxAdultAmount;
                    $itemPriceChild -= $taxChildAmount;
                } else {
                    $taxAmount      = $item->price * $taxTable->rate;
                    $taxAdultAmount = $item->price_adult * $taxTable->rate;
                    $taxChildAmount = $item->price_child * $taxTable->rate;
                }
            }

            // For charge type != per person
            $item->currencyTaxIncl = clone $solidresCurrency;
            $item->currencyTaxExcl = clone $solidresCurrency;
            $item->currencyTaxIncl->setValue($itemPrice + $taxAmount);
            $item->currencyTaxExcl->setValue($itemPrice);
            $item->price_tax_incl = $itemPrice + $taxAmount;
            $item->price_tax_excl = $itemPrice;

            // For adult
            $item->currencyAdultTaxIncl = clone $solidresCurrency;
            $item->currencyAdultTaxExcl = clone $solidresCurrency;
            $item->currencyAdultTaxIncl->setValue($itemPriceAdult + $taxAdultAmount);
            $item->currencyAdultTaxExcl->setValue($itemPriceAdult);
            $item->price_adult_tax_incl = $itemPriceAdult + $taxAdultAmount;
            $item->price_adult_tax_excl = $itemPriceAdult;

            // For child
            $item->currencyChildTaxIncl = clone $solidresCurrency;
            $item->currencyChildTaxExcl = clone $solidresCurrency;
            $item->currencyChildTaxIncl->setValue($itemPriceChild + $taxChildAmount);
            $item->currencyChildTaxExcl->setValue($itemPriceChild);
            $item->price_child_tax_incl = $itemPriceChild + $taxChildAmount;
            $item->price_child_tax_excl = $itemPriceChild;

            if ($showTaxIncl) {
                $item->currency      = $item->currencyTaxIncl;
                $item->currencyAdult = $item->currencyAdultTaxIncl;
                $item->currencyChild = $item->currencyChildTaxIncl;
            } else {
                $item->currency      = $item->currencyTaxExcl;
                $item->currencyAdult = $item->currencyAdultTaxExcl;
                $item->currencyChild = $item->currencyChildTaxExcl;
            }
        }

        return $item;
    }
}
