<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Joomla\Component\Solidres\Administrator\Model;

defined('_JEXEC') or die('Restricted access');

use Exception;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Language\Text;
use Joomla\CMS\MVC\Model\ListModel;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Component\Solidres\Administrator\Table\PaymentHistoryTable;
use Joomla\Database\ParameterType;
use Solidres\Config\Config;
use Solidres\Utility\Utility;

class PaymenthistoryModel extends ListModel
{
    protected $filterFormName = 'filter_paymenthistory';

    public function __construct(array $config = [])
    {
        if (empty($config['filter_fields'])) {
            $config['filter_fields'] = [
                'id', 'a.id',
                'reservation_id', 'a.reservation_id',
                'scope',
                'a.scope',
                'payment_type',
                'a.payment_type',
                'payment_date',
                'a.payment_date',
                'payment_method_id',
                'a.payment_method_id',
                'payment_method_txn_id',
                'a.payment_method_txn_id',
                'payment_method_surcharge',
                'a.payment_method_surcharge',
                'payment_method_discount',
                'a.payment_method_discount',
                'payment_amount',
                'a.payment_amount',
                'reservation_code',
            ];
        }

        parent::__construct($config);
    }

    protected function populateState($ordering = 'a.payment_date', $direction = 'desc')
    {
        $value = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search', '', 'string');
        $this->setState('filter.search', $value);

        $value = $this->getUserStateFromRequest(
            $this->context . '.filter.payment_date',
            'filter_payment_date',
            '',
            'user_utc'
        );
        $this->setState('filter.payment_date', $value);

        $value = $this->getUserStateFromRequest(
            $this->context . '.filter.payment_status',
            'filter_payment_status',
            '',
            'string'
        );
        $this->setState('filter.payment_status', $value);

        $value = $this->getUserStateFromRequest(
            $this->context . '.filter.payment_method_id',
            'filter_payment_method_id',
            '',
            'string'
        );
        $this->setState('filter.payment_method_id', $value);

        $value = $this->getUserStateFromRequest(
            $this->context . '.filter.payment_type',
            'filter_payment_type',
            '0',
            'uint'
        );
        $this->setState('filter.payment_type', $value);

        $value = $this->getUserStateFromRequest($this->context . '.filter.scope', 'filter_scope', 0, 'uint');

        if (!is_numeric($value)
            || !in_array((int)$value, [0, 1])
            || ((int)$value && !PluginHelper::isEnabled('solidres', 'experience'))) {
            $value = 0;
        }

        $this->setState('filter.scope', $value);

        parent::populateState($ordering, $direction);
    }

    protected function getStoreId($id = '')
    {
        $id .= ':' . $this->getState('filter.search');
        $id .= ':' . $this->getState('filter.payment_date');
        $id .= ':' . $this->getState('filter.payment_status');
        $id .= ':' . $this->getState('filter.payment_method_id');
        $id .= ':' . $this->getState('filter.reservation_id');
        $id .= ':' . $this->getState('filter.scope');
        $id .= ':' . $this->getState('filter.payment_type');

        return parent::getStoreId($id);
    }

    protected function getListQuery()
    {
        $scope = (int)$this->getState('filter.scope', 0);
        $db    = $this->getDatabase();
        $query = $db->getQuery(true)
            ->select(
                $this->getState(
                    'list.select',
                    'a.id, a.reservation_id, a.scope, a.payment_date, a.payment_status, a.payment_data, a.currency_id, '
                    . 'a.payment_method_id, a.payment_method_txn_id, a.payment_method_surcharge, a.payment_method_discount, a.payment_amount, '
                    . 'a.title, a.description, a.payment_type'
                )
            )
            ->from($db->quoteName('#__sr_payment_history', 'a'))
            ->where('a.scope = ' . $scope);
        $query->select('a2.code AS reservation_code');

        if ($scope) {
            $query->select('a2.experience_id, a2.currency_code')
                ->innerJoin($db->quoteName('#__sr_experience_reservations', 'a2') . ' ON a2.id = a.reservation_id');
        } else {
            $query->select('a2.reservation_asset_id, a2.currency_code')
                ->innerJoin($db->quoteName('#__sr_reservations', 'a2') . ' ON a2.id = a.reservation_id');
        }

        $search = $this->getState('filter.search');

        if (!empty($search)) {
            if (stripos($search, 'id:') === 0) {
                $query->where('a.id = ' . (int)substr($search, 3));
            } elseif (stripos($search, 'reservation:') === 0) {
                $value = substr($search, 12);

                if (is_numeric($value) && strpos($value, '0') !== 0) {
                    $query->where('a2.id = ' . (int)$value);
                } else {
                    $query->where('a2.code = ' . $db->quote($value));
                }
            } else {
                $search = $db->quote('%' . $db->escape($search, true) . '%');
                $query->where(
                    '(a.payment_method_id LIKE ' . $search
                    . ' OR a.payment_method_txn_id LIKE ' . $search
                    . ' OR a2.code LIKE ' . $search . ')'
                );
            }
        }

        $date = $this->getState('filter.payment_date');

        if (!empty($date) && $date != $db->getNullDate()) {
            try {
                $date = Factory::getDate($date);
                $query->where('DATE(a.payment_date) = ' . $db->quote($date->format('Y-m-d')));
            } catch (Exception $e) {
            }
        }

        $status = $this->getState('filter.payment_status');

        if (is_numeric($status)) {
            $query->where('a.payment_status = ' . (int)$status);
        }

        $element = $this->getState('filter.payment_method_id');

        if (!empty($element)) {
            $query->where('a.payment_method_id = ' . $db->quote($element));
        }

        if ($reservationId = $this->getState('filter.reservation_id')) {
            $query->where('a.reservation_id = ' . (int)$reservationId);
        }

        $query->where('a.payment_type = ' . (int)$this->getState('filter.payment_type', 0));

        $query->select('a3.label AS payment_status_label, a3.color_code AS payment_status_color')
            ->leftJoin(
                $db->quoteName(
                    '#__sr_statuses',
                    'a3'
                ) . ' ON a3.code = a.payment_status AND a3.scope = a.scope AND a3.type = 1'
            );

        $app  = Factory::getApplication();
        $user = $app->getIdentity();

        if ($app->isClient('api') && !$user->authorise('core.manage', 'com_solidres')) {
            $userId    = $user->id;
            $joinTable = $scope ? '#__sr_experiences' : '#__sr_reservation_assets';
            $joinKey   = $scope ? 'experience_id' : 'reservation_asset_id';
            $query->join('INNER', $db->quoteName($joinTable, 'a4') . ' ON a4.id = a2.' . $joinKey)
                ->join('INNER', $db->quoteName('#__sr_customers', 'a5') . ' ON a5.id = a4.partner_id')
                ->where('a5.user_id = :userId')
                ->bind(':userId', $userId, ParameterType::INTEGER);
        }

        $ordering  = $this->getState('list.ordering', 'a.payment_date');
        $direction = $this->getState('list.direction', 'desc');
        $query->order($db->escape($ordering) . ' ' . $db->escape($direction));

        return $query;
    }

    public function getFilterForm($data = [], $loadData = true)
    {
        if ($form = parent::getFilterForm($data, $loadData)) {
            $form->setFieldAttribute('payment_status', 'scope', $form->getValue('scope', 'filter', 0), 'filter');
        }

        return $form;
    }

    public function isOwner($scope, $reservationId)
    {
        $app  = Factory::getApplication();
        $user = $app->getIdentity();

        if ($app->isClient('administrator')) {
            return $user->authorise('core.admin', 'com_solidres') ? true : false;
        }

        if (!$user->id) {
            return false;
        }

        $db    = $this->getDatabase();
        $query = $db->getQuery(true)
            ->select('COUNT(a.id)');

        if ($scope) {
            $query->from($db->quoteName('#__sr_experiences', 'a'))
                ->innerJoin($db->quoteName('#__sr_experience_reservations', 'a2') . ' ON a2.experience_id = a.id');

            $query->innerJoin($db->quoteName('#__sr_customers', 'a3') . ' ON a3.id = a.partner_id')
                ->where('a2.id = ' . (int)$reservationId)
                ->where('a3.user_id = ' . (int)$user->id);
        } else {
            $query->from($db->quoteName('#__sr_reservation_assets', 'a'))
                ->innerJoin($db->quoteName('#__sr_reservations', 'a2') . ' ON a2.reservation_asset_id = a.id');

            $properties = Utility::getPropertiesByPartner();

            if (is_array($properties) && count($properties) > 0) {
                $query->where('a.id IN (' . join(',', array_keys($properties)) . ')');
            }
        }

        $db->setQuery($query);

        return $db->loadResult() ? true : false;
    }

    protected function loadReservationData($paymentHistoryTable)
    {
        $db             = $this->getDatabase();
        $reservationId  = (int)$paymentHistoryTable->reservation_id;
        $paymentType    = (int)$paymentHistoryTable->payment_type;
        $solidresParams = ComponentHelper::getParams('com_solidres');

        if ($paymentHistoryTable->scope) {
            if (!PluginHelper::isEnabled('solidres', 'experience')) {
                return true;
            }

            $reservationTable = $this->getTable('ExpReservation', 'Administrator');

            if ($reservationTable->load($reservationId)) {
                $confirmationPaymentState = (int)$solidresParams->get('exp_payment_confirm_state', 1);
                $totalDiscount            = (float)$reservationTable->total_discount;
                $query                    = $db->getQuery(true)
                    ->select(
                        'SUM(a.payment_amount + a.payment_method_surcharge - ' . $totalDiscount . ' - a.payment_method_discount)'
                    )
                    ->from($db->quoteName('#__sr_payment_history', 'a'))
                    ->where('a.scope = 1')
                    ->where('a.payment_type = ' . $paymentType)
                    ->where('a.payment_status = ' . $confirmationPaymentState)
                    ->where('a.reservation_id = ' . $reservationId);
                $db->setQuery($query);
                $totalPaid = (float)($db->loadResult() ?: 0.00);
                $reservationTable->set('total_paid', $totalPaid);
                $grandTotal = (float)$reservationTable->total_price + ((float)$reservationTable->payment_method_surcharge - (float)$reservationTable->payment_method_discount - (float)$reservationTable->total_discount);

                if ($totalPaid >= $grandTotal) {
                    $reservationTable->set('payment_status', $confirmationPaymentState);
                }

                $reservationTable->store();

                return $reservationTable->getProperties();
            }
        } else {
            $reservationTable = $this->getTable('Reservation', 'Administrator');

            if ($reservationTable->load($reservationId)) {
                $confirmationPaymentState = (int)$solidresParams->get('confirm_payment_state', 1);
                $query                    = $db->getQuery(true)
                    ->select('SUM(a.payment_amount + a.payment_method_surcharge - a.payment_method_discount)')
                    ->from($db->quoteName('#__sr_payment_history', 'a'))
                    ->where('a.scope = 0')
                    ->where('a.payment_type = ' . $paymentType)
                    ->where('a.payment_status = ' . $confirmationPaymentState)
                    ->where('a.reservation_id = ' . $reservationId);
                $db->setQuery($query);
                $totalPaid = (float)($db->loadResult() ?: 0.00);

                // This is for commissions, let ignore updating total_paid for reservation
                if (0 == $paymentType) {
                    $reservationTable->set('total_paid', $totalPaid);
                }

                $reservationData = $reservationTable->getProperties();

                if ($reservationData['discount_pre_tax']) {
                    $grandTotal = (float)$reservationData['total_price_tax_excl'] - (float)$reservationData['total_discount'] + (float)$reservationData['tax_amount'] + (float)$reservationData['total_extra_price_tax_incl'];
                } else {
                    $grandTotal = (float)$reservationData['total_price_tax_excl'] + (float)$reservationData['tax_amount'] - (float)$reservationData['total_discount'] + (float)$reservationData['total_extra_price_tax_incl'];
                }

                $grandTotal                    += (float)$reservationData['tourist_tax_amount'];
                $grandTotal                    += (float)$reservationData['total_fee'];
                $reservationData['grandTotal'] = (float)$grandTotal;
                $reservationData['totalDue']   = (float)$reservationData['grandTotal'] - (float)$reservationData['total_paid'];

                if ($totalPaid >= $grandTotal) {
                    $reservationTable->set('payment_status', $confirmationPaymentState);
                }

                $reservationTable->store();

                return $reservationData;
            }
        }

        return true;
    }

    public function save(array $data, $checkOwner = true)
    {
        if (!isset($data['scope'])
            || !isset($data['reservation_id'])
            || ($checkOwner && !$this->isOwner($data['scope'], $data['reservation_id']))
        ) {
            $this->setError(Text::_('JERROR_ALERTNOAUTHOR'));

            return false;
        }

        $table = $this->getTable('PaymentHistory', 'Administrator');

        if (!$table->bind($data)
            || !$table->check()
            || !$table->store(true)
        ) {
            $this->setError($table->getError());

            return false;
        }

        return $this->loadReservationData($table);
    }

    public function getForm($data = [])
    {
        Form::addFormPath(JPATH_ADMINISTRATOR . '/components/com_solidres/forms');
        Form::addFieldPath(JPATH_ADMINISTRATOR . '/components/com_solidres/models/fields');

        $paymentType = $this->getState('filter.payment_type', 0);
        $form        = $this->loadForm(
            'com_solidres.paymenthistory' . $paymentType,
            'paymenthistory',
            ['control' => 'paymentHistoryForm' . $paymentType, 'load_data' => false]
        );

        if (empty($form)) {
            return false;
        }

        if ($data) {
            $form->bind($data);
        }

        return $form;
    }

    public function getPaymentElements($scope, $scopeId)
    {
        $db    = $this->getDatabase();
        $query = $db->getQuery(true)
            ->clear()
            ->select('a.element')
            ->from($db->quoteName('#__extensions', 'a'))
            ->where('a.enabled = 1')
            ->where('a.type = ' . $db->quote('plugin'))
            ->where('a.folder = ' . $db->quote($scope ? 'experiencepayment' : 'solidrespayment'));
        $db->setQuery($query);
        $results = [];

        if ($elements = $db->loadColumn()) {
            $config   = new Config(['scope_id' => $scopeId]);
            $language = Factory::getApplication()->getLanguage();

            foreach ($elements as $element) {
                if ($scope) {
                    $group     = 'experiencepayment';
                    $nameKey   = 'experience/payments/' . $element . '_enabled';
                    $stringKey = 'PLG_EXPERIENCEPAYMENT_' . strtoupper($element) . '_LABEL';
                } else {
                    $group     = 'solidrespayment';
                    $nameKey   = 'payments/' . $element . '/' . $element . '_enabled';
                    $stringKey = 'SR_PAYMENT_METHOD_' . strtoupper($element);
                }

                if ($config->get($nameKey)) {
                    $language->load('plg_' . $group . '_' . $element, JPATH_PLUGINS . '/' . $group . '/' . $element);
                    $results[$element] = Text::_($stringKey);
                }
            }
        }

        return $results;
    }

    public function delete($id)
    {
        /** @var $table PaymentHistoryTable */
        $table = $this->getTable('PaymentHistory', 'Administrator');
        $id    = (int)$id;

        if ($id < 0
            || !$table->load($id)
            || !$this->isOwner($table->scope, $table->reservation_id)) {
            $this->setError(Text::_('JERROR_ALERTNOAUTHOR'));

            return false;
        }

        if (!$table->delete($id)) {
            $this->setError($table->getError());

            return false;
        }

        return $this->loadReservationData($table);
    }
}
