<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Joomla\Component\Solidres\Administrator\Model;

defined('_JEXEC') or die;

use Joomla\CMS\Application\ApplicationHelper;
use Joomla\CMS\Component\ComponentHelper;
use Joomla\CMS\Event\Model\AfterSaveEvent;
use Joomla\CMS\Event\Model\BeforeSaveEvent;
use Joomla\CMS\Event\Model\PrepareDataEvent;
use Joomla\CMS\Event\Model\PrepareFormEvent;
use Joomla\CMS\Factory;
use Joomla\CMS\Form\Form;
use Joomla\CMS\Plugin\PluginHelper;
use Joomla\Component\Solidres\Administrator\Table\RoomTypeTable;
use Solidres\Currency\Currency;
use Solidres\Event\SolidresBaseEvent;
use Solidres\Event\SolidresResultBaseEvent;
use Solidres\Media\ImageUploaderHelper;
use Solidres\Media\ImageUploaderTrait;
use Solidres\MVC\FactoryTrait;
use Solidres\MVC\Model\AdminModel;
use Solidres\RoomType\RoomType;
use Solidres\Utility\Utility;
use Throwable;

class RoomtypeModel extends AdminModel
{
    private static $propertiesCache = [];

    private static $taxesCache = [];

    use ImageUploaderTrait;

    use FactoryTrait;

    public function __construct($config = [])
    {
        parent::__construct($config);

        $this->event_after_delete  = 'onRoomTypeAfterDelete';
        $this->event_after_save    = 'onRoomTypeAfterSave';
        $this->event_before_delete = 'onRoomTypeBeforeDelete';
        $this->event_before_save   = 'onRoomTypeBeforeSave';
        $this->event_change_state  = 'onRoomTypeChangeState';
    }

    public function getForm($data = [], $loadData = true)
    {
        $form = parent::getForm($data, $loadData);

        if (empty($form)) {
            return false;
        }

        // Determine correct permissions to check.
        if ($this->getState('roomtype.id')) {
            // Existing record. Can only edit in selected categories.
            $form->setFieldAttribute('reservation_asset_id', 'action', 'core.edit');
        } else {
            // New record. Can only create in selected categories.
            $form->setFieldAttribute('reservation_asset_id', 'action', 'core.create');
        }

        // When Complex Tariff plugin is in use, Standard rate will be optional
        if (PluginHelper::isEnabled('solidres', 'complextariff')) {
            $form->setFieldAttribute('default_tariff', 'required', false);
        }

        return $form;
    }

    public function save($data)
    {
        /** @var RoomTypeTable $table */
        $table = $this->getTable();
        $pk    = (!empty($data['id'])) ? $data['id'] : (int)$this->getState($this->getName() . '.id');
        $isNew = true;

        // Include the content plugins for the on save events.
        PluginHelper::importPlugin('extension');
        PluginHelper::importPlugin('solidres');

        // Load the row if saving an existing record.
        if ($pk > 0) {
            $table->load($pk);
            $isNew = false;
        }

        // Bind the data.
        if (!$table->bind($data)) {
            $this->setError($table->getError());

            return false;
        }

        // Prepare the row for saving
        $this->prepareTable($table);

        // Check the data.
        if (!$table->check()) {
            $this->setError($table->getError());

            return false;
        }

        $app = Factory::getApplication();

        // Trigger the onContentBeforeSave event.
        $dispatcher = $app->getDispatcher();
        $result     = $dispatcher->dispatch(
            $this->event_before_save,
            new BeforeSaveEvent(
                $this->event_before_save, [
                'context' => 'com_solidres.roomtype',
                'data'    => $data,
                'subject' => $table,
                'isNew'   => $isNew,
            ]
            )
        )
            ->getArgument('result', []);

        if (in_array(false, $result, true)) {
            $this->setError($table->getError());

            return false;
        }

        // Store the data.
        if (!$table->store()) {
            $this->setError($table->getError());

            return false;
        }

        // Clean the cache.
        $cache = Factory::getCache($this->option);
        $cache->clean();

        try {
            $this->uploadMedia($table->id);
        } catch (Throwable $e) {
        }

        $dbo        = $this->getDatabase();
        $query      = $dbo->createQuery();
        $srRoomType = new RoomType();
        $nullDate   = substr($dbo->getNullDate(), 0, 10);
        $context    = 'com_solidres.edit.roomtype';

        // ==  Processing tariff/prices == //
        // Check the current currency_id of this roomtype's reservation asset.
        $query->select('currency_id');
        $query->from($dbo->quoteName('#__sr_reservation_assets'));
        $query->where('id = ' . (int)$data['reservation_asset_id']);
        $currencyId = $dbo->setQuery($query)->loadResult();

        // Store the default tariff
        if (isset($data['default_tariff']) && is_array($data['default_tariff'])) {
            $tariffId   = key($data['default_tariff']);
            $tariffData = [
                'id'                => $tariffId > 0 ? $tariffId : null,
                'currency_id'       => $currencyId,
                'customer_group_id' => null,
                'valid_from'        => $nullDate,
                'valid_to'          => $nullDate,
                'room_type_id'      => $table->id,
                'title'             => $data['standard_tariff_title'],
                'description'       => $data['standard_tariff_description'],
                'd_min'             => null,
                'd_max'             => null,
                'p_min'             => null,
                'p_max'             => null,
                'type'              => 0, // Default is per room per night
                'limit_checkin'     => '',
                'state'             => 1,
            ];

            foreach ($data['default_tariff'][$tariffId] as $detailId => $detailInfo) {
                foreach ($detailInfo as $day => $price) {
                    $tariffData['details']['per_room'][$day]['id']         = $detailId > 0 ? $detailId : null;
                    $tariffData['details']['per_room'][$day]['price']      = $price;
                    $tariffData['details']['per_room'][$day]['w_day']      = $day;
                    $tariffData['details']['per_room'][$day]['guest_type'] = null;
                    $tariffData['details']['per_room'][$day]['from_age']   = null;
                    $tariffData['details']['per_room'][$day]['to_age']     = null;
                }
            }

            $tariffModel = $this->createModel('Tariff');
            $tariffModel->save($tariffData);
        }

        // ==  Processing tariff/prices == //

        $query->clear();
        $query->delete($dbo->quoteName('#__sr_room_type_coupon_xref'))->where(
            'room_type_id = ' . $dbo->quote($table->id)
        );
        $dbo->setQuery($query);
        $result = $dbo->execute();

        if (!$result) {
            $app->enqueueMessage(
                'plgExtensionSolidres::onRoomTypeAfterSave: Delete from ' . $dbo->quoteName(
                    '#__sr_room_type_coupon_xref'
                ) . ' failure',
                'warning'
            );
        }

        if (isset($data['coupon_id']) && count($data['coupon_id'])) {
            foreach ($data['coupon_id'] as $value) {
                $srRoomType->storeCoupon($table->id, $value);
            }
        }

        $query->clear();
        $query->delete($dbo->quoteName('#__sr_room_type_extra_xref'))->where(
            'room_type_id = ' . $dbo->quote($table->id)
        );
        $dbo->setQuery($query);
        $result = $dbo->execute();

        if (!$result) {
            $app->enqueueMessage(
                'plgExtensionSolidres::onRoomTypeAfterSave: Delete from ' . $dbo->quoteName(
                    '#__sr_room_type_extra_xref'
                ) . ' failure',
                'warning'
            );
        }

        if (isset($data['extra_id']) && count($data['extra_id'])) {
            foreach ($data['extra_id'] as $value) {
                $srRoomType->storeExtra($table->id, $value);
            }
        }

        if (isset($data['rooms']) && count($data['rooms'])) {
            foreach ($data['rooms'] as $value) {
                if ($value['id'] == 'new' && !empty($value['label'])) {
                    $srRoomType->storeRoom($table->id, $value['label']);
                } elseif ($value['id'] > 0 && !empty($value['label'])) {
                    $srRoomType->storeRoom($table->id, $value['label'], $value['id']);
                }
            }
        }

        // Process extra fields
        if ($table->id
            && $result
            && isset($data['roomtype_custom_fields'])
            && (count($data['roomtype_custom_fields']))
        ) {
            $query->clear();
            $query->delete()->from($dbo->quoteName('#__sr_room_type_fields'));
            $query->where('room_type_id = ' . $table->id);
            $query->where("field_key LIKE 'roomtype_custom_fields.%'");
            $dbo->setQuery($query)
                ->execute();
            $tuples = [];
            $order  = 1;

            foreach ($data['roomtype_custom_fields'] as $k => $v) {
                $tuples[] = '(' . $table->id . ', ' . $dbo->quote('roomtype_custom_fields.' . $k) . ', ' . $dbo->quote(
                        $v
                    ) . ', ' . $order++ . ')';
            }

            $dbo->setQuery(
                'INSERT INTO ' . $dbo->quoteName('#__sr_room_type_fields') . ' VALUES ' . implode(', ', $tuples)
            )
                ->execute();
        }

        // If this room type is a result of copying, copy complex tariffs
        $isCopying = $app->getUserState($context . '.is_copying_room_type', false);

        if ($isCopying) {
            $isCopyingFrom = $app->getUserState($context . '.is_copying_room_type_from', false);
            $query->clear();
            $query->select('id')->from($dbo->quoteName('#__sr_tariffs'))->where('room_type_id = ' . (int)$isCopyingFrom)
                ->where('valid_from != ' . $dbo->quote('0000-00-00'))
                ->where('valid_to != ' . $dbo->quote('0000-00-00'));

            $copiedTariffs = $dbo->setQuery($query)->loadObjectList();
            foreach ($copiedTariffs as $copiedTariff) {
                $tariffModel                      = $this->createModel('Tariff');
                $copiedTariffData                 = get_object_vars($tariffModel->getItem($copiedTariff->id));
                $copiedTariffData['room_type_id'] = $table->id;
                $copiedTariffData['id']           = null;

                if ($copiedTariffData['mode'] == 1) {
                    $copiedTariffData['details'] = json_decode(json_encode($copiedTariffData['details']), true);

                    if (isset($copiedTariffData['details']) && is_array($copiedTariffData['details'])) {
                        foreach ($copiedTariffData['details'] as $type => $months) {
                            $flatten = [];
                            foreach ($months as $month => $dates) {
                                $flatten = array_merge($flatten, $dates);
                            }

                            $copiedTariffData['details'][$type] = $flatten;
                        }
                    }
                }

                foreach ($copiedTariffData['details'] as $tariffType => &$details) {
                    foreach ($details as &$detail) {
                        $detail       = (array)$detail;
                        $detail['id'] = null;
                    }
                }
                $tariffModel->save($copiedTariffData);
            }

            $app->setUserState($context . '.is_copying_room_type', false);
            $app->setUserState($context . '.is_copying_room_type_from', false);
        }

        // Trigger the onContentAfterSave event.
        $dispatcher->dispatch(
                $this->event_after_save,
                new AfterSaveEvent(
                    $this->event_after_save,
                    [
                        'context' => 'com_solidres.roomtype',
                        'data'    => $data,
                        'subject' => $table,
                        'isNew'   => $isNew,
                    ]
                )
            );
        $this->setState($this->getName() . '.id', $table->id);
        $this->setState($this->getName() . '.new', $isNew);

        return true;
    }

    public function getTable($name = 'RoomType', $prefix = 'Administrator', $options = [])
    {
        return parent::getTable($name, $prefix, $options);
    }

    protected function prepareTable($table)
    {
        $date = Factory::getDate();
        $user = Factory::getApplication()->getIdentity();

        $table->name  = htmlspecialchars_decode($table->name, ENT_QUOTES);
        $table->alias = ApplicationHelper::stringURLSafe($table->alias);

        if (empty($table->alias)) {
            $table->alias = ApplicationHelper::stringURLSafe($table->name);
        }

        if (empty($table->params)) {
            $table->params = '';
        }

        if (empty($table->id)) {
            $table->created_date = $date->toSql();

            // Set ordering to the last item if not set
            if (empty($table->ordering)) {
                $db    = $this->getDatabase();
                $query = $db->getQuery(true);
                $query->clear();
                $query->select('MAX(ordering)')->from($db->quoteName('#__sr_room_types'));
                $db->setQuery($query);
                $max = $db->loadResult();

                $table->ordering = $max + 1;
            }
        } else {
            $table->modified_date = $date->toSql();
            $table->modified_by   = $user->get('id');
        }
    }

    public function delete(&$pks)
    {
        PluginHelper::importPlugin('solidres');

        return parent::delete($pks);
    }

    public function getItem($pk = null)
    {
        $item             = parent::getItem($pk);
        $solidresConfig   = ComponentHelper::getParams('com_solidres');
        $showPriceWithTax = $solidresConfig->get('show_price_with_tax', 0);
        $numberOfDecimals = $solidresConfig->get('number_decimal_points', 2);

        if ($item->id) {
            $app        = Factory::getApplication();
            $dbo        = $this->getDatabase();
            $query      = $dbo->getQuery(true);
            $nullDate   = substr($dbo->getNullDate(), 0, 10);
            $minTariff  = 0;
            $mvcFactory = $app->bootComponent('com_solidres')->getMVCFactory();

            if (!isset(self::$propertiesCache[$item->reservation_asset_id])) {
                $tableRA = $mvcFactory->createTable('ReservationAsset', 'Administrator');
                $tableRA->load($item->reservation_asset_id);

                if (isset($tableRA->params) && is_string($tableRA->params)) {
                    $tableRA->params_decoded = json_decode($tableRA->params, true);
                }

                self::$propertiesCache[$item->reservation_asset_id] = $tableRA;
            }

            $solidresCurrency      = new Currency(0, self::$propertiesCache[$item->reservation_asset_id]->currency_id);
            $assetPriceIncludesTax = self::$propertiesCache[$item->reservation_asset_id]->price_includes_tax;
            $item->currency_id     = self::$propertiesCache[$item->reservation_asset_id]->currency_id;
            $item->tax_id          = self::$propertiesCache[$item->reservation_asset_id]->tax_id;
            if (isset(self::$propertiesCache[$item->reservation_asset_id]->params_decoded)) {
                $item->property_is_apartment = false;
                if (isset(self::$propertiesCache[$item->reservation_asset_id]->params_decoded['is_apartment'])) {
                    $item->property_is_apartment = (bool)self::$propertiesCache[$item->reservation_asset_id]->params_decoded['is_apartment'];
                }
            }

            $assetId             = $item->reservation_asset_id;
            $assetAlias          = self::$propertiesCache[$assetId]->alias;
            $isApartment         = $item->property_is_apartment ?? false;
            $item->property_slug = $assetId . ($isApartment ? '-apartment' : '') . ':' . $assetAlias;
            $item->slug          = $item->id . ':' . $item->alias;

            // Get the advertised price for front end usage
            $advertisedPrice      = (float) ($item->params['advertised_price'] ?? 0);
            $skipFindingMinTariff = false;
            if ($advertisedPrice > 0) {
                $skipFindingMinTariff = true;
                $minTariff            = $advertisedPrice;
            }

            // Load the standard tariff
            $query->select('p.*, c.currency_code, c.currency_name');
            $query->from($dbo->quoteName('#__sr_tariffs') . ' as p');
            $query->join('left', $dbo->quoteName('#__sr_currencies') . ' as c ON c.id = p.currency_id');
            $query->where('room_type_id = ' . (empty($item->id) ? 0 : (int)$item->id));
            $query->where('valid_from = ' . $dbo->quote($nullDate));
            $query->where('valid_to = ' . $dbo->quote($nullDate));

            $item->default_tariff = $dbo->setQuery($query)->loadObject();

            if (isset($item->default_tariff)) {
                $query->clear();
                $query->select('id, tariff_id, price, w_day, guest_type, from_age, to_age');
                $query->from($dbo->quoteName('#__sr_tariff_details'));
                $query->where('tariff_id = ' . (int)$item->default_tariff->id);
                $query->order('w_day ASC');
                $item->default_tariff->details = $dbo->setQuery($query)->loadObjectList();

                foreach ($item->default_tariff->details as $tariff) {
                    if (!$skipFindingMinTariff) {
                        if ($minTariff == 0) {
                            $minTariff = $tariff->price;
                        }
                        if ($minTariff > $tariff->price) {
                            $minTariff = $tariff->price;
                        }
                    }

                    $tariff->price = round($tariff->price, $numberOfDecimals);
                }
            }

            $imposedTaxTypes = [];
            if (!empty($item->tax_id)) {
                if (!isset(self::$taxesCache[$item->tax_id])) {
                    $taxModel                        = $mvcFactory->createModel(
                        'Tax',
                        'Administrator',
                        ['ignore_request' => true]
                    );
                    self::$taxesCache[$item->tax_id] = $taxModel->getItem($item->tax_id);
                }

                $imposedTaxTypes[] = self::$taxesCache[$item->tax_id];
            }

            $taxAmount = 0;

            foreach ($imposedTaxTypes as $taxType) {
                if ($assetPriceIncludesTax == 0) {
                    $taxAmount = $minTariff * $taxType->rate;
                } else {
                    $taxAmount = $minTariff - ($minTariff / (1 + $taxType->rate));
                    $minTariff -= $taxAmount;
                }
            }

            $solidresCurrency->setValue($showPriceWithTax ? ($minTariff + $taxAmount) : $minTariff);

            $item->minTariff = $solidresCurrency;

            $query->clear();
            $query->select('a.id, a.label');
            $query->from($dbo->quoteName('#__sr_rooms') . ' a');
            $query->where('room_type_id = ' . (empty($item->id) ? 0 : (int)$item->id));
            $dbo->setQuery($query);
            $item->roomList = $dbo->loadObjectList();

            // Load media
            $item->media = ImageUploaderHelper::getData($item->id, 'room_type');

            // Custom fields
            $item->roomtype_custom_fields = [];
            $db    = $this->getDatabase();
            $query = $db->createQuery();
            $query->select('field_key, field_value')
                ->from($db->quoteName('#__sr_room_type_fields'))
                ->where('room_type_id = ' . (int)$item->id)
                ->where("field_key LIKE 'roomtype_custom_fields.%'");
            $db->setQuery($query);

            foreach ($db->loadRowList() as $v) {
                $k                                = str_replace('roomtype_custom_fields.', '', $v[0]);
                $item->roomtype_custom_fields[$k] = $v[1];
            }
        }

        return $item;
    }

    public function countRooms($id)
    {
        $dbo   = $this->getDatabase();
        $query = $dbo->getQuery(true);

        $query->select('COUNT(*)')
            ->from($dbo->quoteName('#__sr_rooms'))
            ->where('room_type_id = ' . $dbo->quote($id));

        return $dbo->setQuery($query)->loadResult();
    }

    protected function canDelete($record)
    {
        $app  = Factory::getApplication();
        $user = $app->getIdentity();

        if ($app->isClient('api')) {
            // Authorization has already checked from the ApiController
            return true;
        }

        if ($app->isClient('administrator')) {
            return parent::canDelete($record);
        }

        return Utility::isAssetPartner($user->id, $record->reservation_asset_id);
    }

    protected function canEditState($record)
    {
        $app  = Factory::getApplication();
        $user = $app->getIdentity();

        if ($app->isClient('api')) {
            // Authorization has already checked from the ApiController
            return true;
        }

        if ($app->isClient('administrator')) {
            return parent::canEditState($record);
        } else {
            return Utility::isAssetPartner($user->get('id'), $record->reservation_asset_id);
        }
    }

    protected function loadFormData()
    {
        // Check the session for previously entered form data.
        $data = Factory::getApplication()->getUserState('com_solidres.edit.roomtype.data', []);

        if (empty($data)) {
            $data                              = $this->getItem();
            $data->standard_tariff_title       = $data->default_tariff->title ?? '';
            $data->standard_tariff_description = $data->default_tariff->description ?? '';
        }

        // Get the dispatcher and load the users plugins.
        PluginHelper::importPlugin('solidres');

        // Trigger the data preparation event.
        SolidresBaseEvent::getDispatcher()
            ->dispatch(
                'onRoomTypePrepareData',
                new PrepareDataEvent('onRoomTypePrepareData', ['context' => 'com_solidres.roomtype', 'data' => $data])
            );

        return $data;
    }

    protected function preprocessForm(Form $form, $data, $group = 'extension')
    {
        // Import the appropriate plugin group.
        PluginHelper::importPlugin($group);
        PluginHelper::importPlugin('solidres');

        // Trigger the form preparation event.
        SolidresBaseEvent::getDispatcher()
            ->dispatch(
                'onRoomTypePrepareForm',
                new PrepareFormEvent('onRoomTypePrepareForm', ['subject' => $form, 'data' => $data])
            );
    }

    protected function getReorderConditions($table = null)
    {
        $condition   = [];
        $condition[] = 'reservation_asset_id = ' . (int)$table->reservation_asset_id;

        return $condition;
    }
}
