<?php
/**
 * ------------------------------------------------------------------------
 * SOLIDRES - Accommodation booking extension for Joomla
 * ------------------------------------------------------------------------
 * @author    Solidres Team <contact@solidres.com>
 * @website   https://www.solidres.com
 * @copyright Copyright (C) 2013 Solidres. All Rights Reserved.
 * @license   GNU General Public License version 3, or later
 * ------------------------------------------------------------------------
 */

namespace Joomla\Component\Solidres\Site\Helper;

defined('_JEXEC') or die;

use Joomla\CMS\Application\CMSApplicationInterface;
use Joomla\CMS\Factory;
use Joomla\CMS\User\User;
use Joomla\Component\Solidres\Administrator\Table\WishListTable;
use Joomla\Database\DatabaseDriver;
use Joomla\Database\DatabaseInterface;
use Joomla\Session\SessionInterface;

class WishListHelper
{
    public User $user;

    public DatabaseDriver $db;

    public CMSApplicationInterface $app;

    protected SessionInterface $session;

    protected static array $instances = [];

    protected string $scope;

    protected function __construct(string $scope)
    {
        $this->app     = Factory::getApplication();
        $this->user    = $this->app->getIdentity();
        $this->db      = Factory::getContainer()->get(DatabaseInterface::class);
        $this->session = $this->app->getSession();
        $this->scope   = $scope;
    }

    public static function getInstance($scope = 'reservation_asset'): WishListHelper
    {
        if (!isset(self::$instances[$scope])) {
            self::$instances[$scope] = new WishListHelper($scope);
        }

        return self::$instances[$scope];
    }

    public function add($scopeId, $history = null)
    {
        $scope  = $this->scope . '.' . $scopeId;
        $userId = (int)$this->user->id;

        if (is_array($history) || is_object($history)) {
            $history = json_encode($history);
        }

        $data = [
            'id'            => $scopeId,
            'user_id'       => $userId,
            'created_date'  => Factory::getDate()->toSql(),
            'modified_date' => null,
            'history'       => $history,
            'scope'         => $scope,
        ];

        if ($userId > 0) {
            $table = $this->getTable();

            if (!$table->load(['user_id' => $userId, 'scope' => $scope])) {
                $table->bind(['user_id' => $userId, 'scope' => $scope]);
            }

            $table->history = $history;

            if ($table->store()) {
                $data = array_merge($data, $table->getProperties());
            }
        }

        $this->session->set($scope, $data);

        return $this;
    }

    public function clear($scopeId = 0)
    {
        $scope = $this->scope;

        if ($scopeId > 0) {
            $scope .= '.' . $scopeId;
        }

        $this->session->clear($scope);
        $userId = (int)$this->user->id;

        if ($userId > 0) {
            $query = $this->db->getQuery(true)
                ->delete($this->db->quoteName('#__sr_wishlist'))
                ->where('user_id = ' . $userId . ' AND scope LIKE ' . $this->db->quote($scope . '%'));
            $this->db->setQuery($query)
                ->execute();
        }

        return $this;
    }

    public function getTable(): WishListTable
    {
        return $this->app
            ->bootComponent('com_solidres')
            ->getMVCFactory()
            ->createTable('WishList', 'Administrator');
    }

    public function load($scopeId = 0)
    {
        $userId   = (int)$this->user->id;
        $wishList = [];

        if ($items = $this->session->get($this->scope, [])) {
            // Force convert to array
            foreach ($items as $k => $item) {
                if (!empty($item)) {
                    $wishList[(int)$k] = (array)$item;
                }
            }
        }

        if ($userId > 0) {
            $query = $this->db->getQuery(true)
                ->select('w.id, w.user_id, w.scope, w.created_date, w.modified_date')
                ->from($this->db->quoteName('#__sr_wishlist', 'w'))
                ->where('w.user_id = ' . $userId . ' AND w.scope LIKE ' . $this->db->quote($this->scope . '%'));
            $this->db->setQuery($query);

            if ($items = $this->db->loadObjectList()) {
                foreach ($items as $item) {
                    preg_match('/([0-9]+)$/', $item->scope, $matches);

                    if (!empty($matches[1])) {
                        $wishList[(int)$matches[1]] = (array)$item;
                    }
                }
            }
        }

        return $scopeId > 0 ? @$wishList[$scopeId] : $wishList;
    }
}
